package org.mule.weave.v2.runtime.core.functions

import org.mule.weave.v2.core.functions.QuaternaryFunctionValue
import org.mule.weave.v2.core.functions.WriteFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.service.Debug
import org.mule.weave.v2.model.service.Info
import org.mule.weave.v2.model.service.Warn
import org.mule.weave.v2.model.service.LogLevel
import org.mule.weave.v2.model.service.LoggingService
import org.mule.weave.v2.model.types.AnyType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values._

object LogFunctionValue extends QuaternaryFunctionValue {

  override val First = ObjectType
  override val Second = StringType
  override val Third = StringType
  override val Forth = AnyType

  override def doExecute(context: First.V, levelValue: Second.V, prefixValue: Third.V, messageValue: Forth.V)(implicit ctx: EvaluationContext): Value[_] = {
    val logger = ctx.serviceManager.loggingService

    val logLevel = LogLevel.fromString(levelValue.evaluate.toString).getOrElse(Debug)

    if (logLevelEnabled(context, logLevel, logger)) {
      val messageValueMat = messageValue.materialize
      val writer = WriteFunctionValue
      val logValue = writer.call(messageValueMat, writer.secondDefaultValue.get.value(), writer.thirdDefaultValue.get.value())
      val messageString: String = StringType.coerce(logValue, this).evaluate.toString
      val message: String = prefixValue.evaluate.toString match {
        case prefix if prefix.nonEmpty => prefix + " - " + messageString
        case _                         => messageString
      }
      logLevel match {
        case Debug => logger.logDebug(message)
        case Info  => logger.logInfo(message)
        case Warn  => logger.logWarn(message)
        case _     => logger.logError(message)
      }
      messageValueMat
    } else {
      messageValue
    }
  }

  private def logLevelEnabled(context: First.V, logLevel: LogLevel, logger: LoggingService)(implicit ctx: EvaluationContext): Boolean = {
    logLevel match {
      case Debug if (!logger.isDebugEnabled()) => false
      case Info if (!logger.isInfoEnabled())   => false
      case _ =>
        val configuration = ctx.serviceManager.loggingConfigurationService.configuration()
        var result: Option[Boolean] = None
        val logCtx = context.evaluate
        val moduleName = logCtx.selectValue(KeyValue("module")).evaluate.toString
        if (logCtx.selectValue(KeyValue("function")) != null) {
          val funName = logCtx.selectValue(KeyValue("function")).evaluate.toString
          result = configuration.canLogFunctionAtLevel(moduleName, funName, logLevel)
        }
        if (result.isEmpty) {
          result = configuration.canLogModuleAtLevel(moduleName, logLevel)
        }
        result.getOrElse(logLevel >= configuration.defaultLogLevel())
    }
  }

}