package org.mule.weave.v2.interpreted.extension

import org.mule.weave.v2.core.util.ObjectValueUtils
import org.mule.weave.v2.interpreted.InterpretedModuleExecutableWeave
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.IntModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.StringModuleOption
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.AbstractScopeAnnotationProcessor
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ScopePhaseAnnotationContext
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.ts
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.WeaveType

import java.nio.charset.Charset
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

class WeaveBasedDataFormatExtensionLoaderService(parsingContextFactory: ParsingContextCreator, resourceResolver: WeaveResourceResolver, moduleLoader: RuntimeModuleNodeCompiler) extends DataFormatExtensionsLoaderService {

  def toModuleOption(name: String, value: WeaveType, kvpDocumentation: Option[String]): ModuleOption = {
    val simpleDescription = value.getDocumentation().orElse(kvpDocumentation).getOrElse("")
    val defaultValue = value.getMetadataConstraint("defaultValue").map(_.value)
    val options = value.getMetadataConstraint("possibleValues").map(_.value.toString.split(",").map(_.trim).toSet)
    value match {
      case _: ts.StringType  => StringModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.map(_.toString).orNull, possibleValues = options.getOrElse(Set()))
      case _: ts.BooleanType => BooleanModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.forall(_.toString.toBoolean))
      case _: ts.NumberType  => IntModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.map(_.toString.toInt).getOrElse(0))
      case rt: ReferenceType => toModuleOption(name, rt.resolveType(), kvpDocumentation)
      case _                 => StringModuleOption(name, simpleDescription = simpleDescription) //TODO what to do here???
    }
  }

  def toSettingsOption(maybeParameter: WeaveType): Map[String, ModuleOption] = {
    maybeParameter match {
      case ot: ts.ObjectType => {
        ot.properties
          .flatMap(prop => {
            TypeHelper.propName(prop).map((propName) => (propName, toModuleOption(propName, prop.value, prop.getDocumentation())))
          })
          .toMap
      }
      case rt: ts.ReferenceType => toSettingsOption(rt.resolveType())
      case it: ts.IntersectionType =>
        TypeHelper.resolveIntersection(it.of) match {
          // TODO Specialize this case
          case otherType if !otherType.isInstanceOf[ts.IntersectionType] => toSettingsOption(otherType)
        }
      case ut: ts.UnionType => {
        ut.of.map(toSettingsOption).reduce(_ ++ _)
      }
    }
  }

  def collectReaderOptions(value: TypeNode): Map[String, ModuleOption] = {
    value
      .resultType()
      .flatMap((weaveType) => {
        val maybeReaderType = TypeHelper.selectProperty(weaveType, "reader")

        val maybeParameter = maybeReaderType
          .collect({
            case ft: FunctionType => ft
          })
          .flatMap((ft) => if (ft.params.size == 3) Some(ft.params(2).wtype) else None)

        maybeParameter.map(toSettingsOption)
      })
      .getOrElse(Map.empty)
  }

  def collectWriterOptions(value: TypeNode): Map[String, ModuleOption] = {
    value
      .resultType()
      .flatMap((weaveType) => {
        val maybeReaderType = TypeHelper.selectProperty(weaveType, "writer")

        val maybeParameter = maybeReaderType
          .collect({
            case ft: FunctionType => ft
          })
          .flatMap((ft) => if (ft.params.size == 2) Some(ft.params(1).wtype) else None)

        maybeParameter.map(toSettingsOption)
      })
      .getOrElse(Map.empty)
  }

  override protected def loadModules()(implicit ctx: EvaluationContext): Seq[DataFormat[_, _]] = {
    val extensions = resourceResolver.resolveAll(Extensions.nameIdentifier)
    extensions.flatMap((extension) => {
      val extensionDescriptorParsingContext: ParsingContext = parsingContextFactory.createParsingContext(Extensions.nameIdentifier, false)

      val compilationResult = WeaveCompiler.compile(extension, extensionDescriptorParsingContext)
      if (compilationResult.noErrors()) {
        val moduleDescriptorCtx = EvaluationContext()
        try {
          val declaredModulesValue = compilationResult.getResult().executable.execute()(moduleDescriptorCtx)
          val declaredModules = ArrayType.coerce(declaredModulesValue)(moduleDescriptorCtx).evaluate(moduleDescriptorCtx).toIterator()

          declaredModules
            .flatMap((descriptor) => {
              val moduleName = NameIdentifier(StringType.coerce(descriptor)(moduleDescriptorCtx).evaluate(moduleDescriptorCtx).toString)
              val moduleParsingContext = parsingContextFactory.createParsingContext(moduleName, true)
              val dfProcessor = new DataFormatExtensionAnnotationProcessor()
              moduleParsingContext.registerAnnotationProcessor(Extensions.dfExtension, dfProcessor)
              val moduleResource = resourceResolver
                .resolveAll(moduleName)
                .headOption
                .getOrElse(throw new WeaveRuntimeException(s"Unable to found resource ${moduleName}", descriptor.location().asInstanceOf[WeaveLocation]))

              val moduleTypeCheck = ModuleParser.parse(ModuleParser.typeCheckPhase(), moduleResource, moduleParsingContext)
              if (moduleTypeCheck.noErrors()) {
                val moduleCompilationResult = WeaveCompiler.runtimeModuleCompilation(moduleTypeCheck.getResult(), moduleParsingContext, moduleLoader)
                val executableModule = moduleCompilationResult.getResult().executable.asInstanceOf[InterpretedModuleExecutableWeave]
                val dataFormats = executableModule.collectVariables(dfProcessor.declaredDataFormats.map(_.name))(moduleDescriptorCtx)
                dataFormats.map((entry) => {
                  val name = entry._1

                  val varIdentifier = dfProcessor.declaredDataFormats.find(_.name == name).get
                  val maybeNode: Option[TypeNode] = moduleTypeCheck.getResult().typeGraph.findNode(varIdentifier)

                  val readerOptions: Map[String, ModuleOption] = maybeNode.map(collectReaderOptions).getOrElse(Map.empty)

                  val writerOptions: Map[String, ModuleOption] = maybeNode.map(collectWriterOptions).getOrElse(Map.empty)

                  val value = ObjectType.coerce(entry._2)(moduleDescriptorCtx)
                  val mimeTypesValue: Seq[MimeType] = ObjectValueUtils
                    .selectStringSeq(value.evaluate(moduleDescriptorCtx), "acceptedMimeTypes")(moduleDescriptorCtx)
                    .map(_.map(MimeType.fromSimpleString))
                    .getOrElse(Seq())

                  val labelValue: Option[String] = ObjectValueUtils
                    .selectString(value.evaluate(moduleDescriptorCtx), "label")(moduleDescriptorCtx)

                  val fileExtensionsValue: Seq[String] = ObjectValueUtils
                    .selectStringSeq(value.evaluate(moduleDescriptorCtx), "fileExtensions")(moduleDescriptorCtx)
                    .getOrElse(Seq())

                  val binaryDataFormatValue: Boolean = ObjectValueUtils
                    .selectBoolean(value.evaluate(moduleDescriptorCtx), "binaryFormat")(moduleDescriptorCtx)
                    .getOrElse(false)

                  val defaultCharsetValue: Option[Charset] = ObjectValueUtils
                    .selectString(value.evaluate(moduleDescriptorCtx), "defaultCharset")(moduleDescriptorCtx)
                    .map(Charset.forName)
                  new WeaveBasedDataFormatProxy(
                    value,
                    () => new WeaveBasedDataFormatSettings(readerOptions),
                    () => new WeaveBasedDataFormatWriterSettings(writerOptions),
                    name,
                    mimeTypesValue,
                    fileExtensionsValue,
                    labelValue,
                    binaryDataFormatValue,
                    defaultCharsetValue)
                })
              } else {
                moduleDescriptorCtx.serviceManager.loggingService.logError(s"Unable to load module `${moduleName}` reasons: ")
                moduleTypeCheck
                  .messages()
                  .errorMessages
                  .foreach((message) => {
                    moduleDescriptorCtx.serviceManager.loggingService.logError(message._2.message + "\n" + message._1.locationString)
                  })
                Seq()
              }
            })
        } finally {
          moduleDescriptorCtx.close()
        }

      } else {
        ctx.serviceManager.loggingService.logError(s"Unable to load descriptor `${extension.url()}` reasons: ")
        compilationResult
          .messages()
          .errorMessages
          .foreach((message) => {
            ctx.serviceManager.loggingService.logError(message._2.message + "\n" + message._1.locationString)
          })
        Seq()
      }
    })
  }

}
object WeaveBasedDataFormatExtensionLoaderService {
  def apply(parsingContextFactory: ParsingContextCreator, resourceResolver: WeaveResourceResolver, moduleLoader: RuntimeModuleNodeCompiler): WeaveBasedDataFormatExtensionLoaderService = {
    new WeaveBasedDataFormatExtensionLoaderService(parsingContextFactory, resourceResolver, moduleLoader)
  }
}

class DataFormatExtensionAnnotationProcessor extends AbstractScopeAnnotationProcessor {

  val declaredDataFormats: ArrayBuffer[NameIdentifier] = mutable.ArrayBuffer[NameIdentifier]()

  override def run(annotatedNode: AstNode, annotation: AnnotationNode, context: ScopePhaseAnnotationContext): Unit = {
    annotatedNode match {
      case vd: VarDirective => {
        declaredDataFormats.+=(vd.variable)
      }
      case _ => //this should never happen
    }
  }
}

object Extensions {
  val nameIdentifier = NameIdentifier("META-INF::dw-extensions")

  val dfExtension = NameIdentifier("dw::extension::DataFormat::DataFormatExtension")
}