package org.mule.weave.v2.runtime.utils

import org.mule.weave.v2.debugger.util.InputElementFactory
import org.mule.weave.v2.interpreted.DefaultRuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.InterpretedMappingExecutableWeave
import org.mule.weave.v2.interpreted.InterpretedModuleExecutableWeave
import org.mule.weave.v2.interpreted.debugger.server.DefaultWeaveDebuggingSession
import org.mule.weave.v2.interpreted.debugger.server.tcp.TcpServerProtocol
import org.mule.weave.v2.interpreted.module.WeaveDataFormat
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.runtime.ExecutableWeaveHelper.buildReaders
import org.mule.weave.v2.runtime.ExecutableWeaveHelper.buildWriter
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.runtime.utils.AnsiColor.red
import org.mule.weave.v2.runtime.utils.AnsiColor.yellow
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceFactory

import java.io.File
import java.io.FileOutputStream
import java.io.OutputStream
import java.util.concurrent.CountDownLatch

object DefaultCustomRunner extends CustomRunner {

  type T = RunnerConfiguration

  override def runnerName(): String = "default"

  def parseArgs(args: Array[String]): Either[RunnerConfiguration, String] = {
    var output: OutputStream = System.out
    var inputs: Seq[(String, File)] = Seq()
    var input: Option[File] = None
    var parameters: Seq[String] = Seq()
    var debug: Boolean = false
    var resource: WeaveResource = null
    var nameIdentifier: NameIdentifier = null
    if (args.isEmpty) {
      Right("Data Weave Name Identifier was not specified.")
    } else {
      val file = new File(args.last)
      if (!file.exists()) {
        nameIdentifier = NameIdentifier(args.last)
        val mayBeUrl = ClassLoaderWeaveResourceResolver().resolve(nameIdentifier)
        mayBeUrl match {
          case Some(value) => {
            resource = value
          }
          case None => {
            return Right(s"Unable to load weave file ${args.last}")
          }
        }
      } else {
        resource = WeaveResourceFactory.fromFile(file)
        val filePath = file.getPath
        val basePath = new File(".").getAbsolutePath
        var path = filePath
        if (filePath.startsWith(basePath)) {
          path = filePath.substring(basePath.length)
        }
        nameIdentifier = NameIdentifier.fromPath(path)
      }

      val configArgs: Array[String] = args.slice(0, args.length - 1)
      var index: Int = 0
      while (index < configArgs.length) {
        configArgs(index) match {
          case "-input" => {
            if (index + 2 < configArgs.length) {
              val input: File = new File(configArgs(index + 2))
              val inputName: String = configArgs(index + 1)
              if (input.exists()) {
                inputs = inputs :+ (inputName, input)
              } else {
                return Right(red(s"Invalid input file $inputName ${input.getAbsolutePath}."))
              }
            } else {
              return Right(red("Invalid amount of arguments on input."))
            }
            index = index + 2
          }
          case "-scenario" => {
            if (index + 1 < configArgs.length) {
              val dir = new File(configArgs(index + 1))
              if (dir.exists()) {
                if (dir.isDirectory) {
                  input = Some(dir)
                } else {
                  return Right(red(s"`scenario` ${dir.getAbsolutePath} is not a directory."))
                }
              } else {
                return Right(red(s"`scenario` ${dir.getAbsolutePath} does not exists."))
              }
            } else {
              return Right(red("Invalid amount of arguments on `scenario`."))
            }
            index = index + 1
          }
          case "-param" | "-p" => {
            if (index + 1 < configArgs.length) {
              val paramValue: String = configArgs(index + 1)
              parameters = parameters :+ paramValue
            } else {
              return Right(red("Invalid amount of arguments on `param`."))
            }
            index = index + 2
          }
          case "-output" => {
            if (index + 1 < configArgs.length) {
              output = new FileOutputStream(new File(configArgs(index + 1)))
            } else {
              return Right(red("Invalid amount of arguments on output."))
            }
            index = index + 1
          }
          case "-debug" => {
            debug = true
          }
          case a => {
            return Right(red(s"Invalid argument $a."))
          }
        }
        index += 1
      }
      Left(RunnerConfiguration(resource, nameIdentifier, input, output, inputs, parameters, debug))
    }
  }

  def createParsingContextFor(configuration: RunnerConfiguration, nameIdentifier: NameIdentifier): ParsingContext = {
    val parsingContext = ParsingContextFactory.createParsingContext(nameIdentifier)
    configuration.inputs.foreach((input) => {
      parsingContext.addImplicitInput(input._1, None)
    })
    contextValues(configuration).foreach((entry) => {
      parsingContext.addImplicitInput(entry._1, None)
    })
    parsingContext
  }

  def run(config: RunnerConfiguration): Unit = {

    val resource = config.resource
    val nameIdentifier = config.nameIdentifier

    val parsingContext = createParsingContextFor(config, nameIdentifier)
    val documentParser = new DocumentParser()
    val parse = documentParser.parse(resource, parsingContext)
    val typeCheckResult: PhaseResult[TypeCheckingResult[_ <: AstNode]] = documentParser.typeCheck(documentParser.scopeCheck(documentParser.canonical(parse, parsingContext), parsingContext), parsingContext)

    val warningMessages = typeCheckResult.warningMessages()
    if (typeCheckResult.hasResult()) {
      if (warningMessages.nonEmpty) {
        warningMessages.foreach((message) => {
          println(yellow(s"[Warning] ${message._2.message}." + s" {${message._1.startPosition.line}:${message._1.startPosition.column}}:\n${message._1.locationString}"))
        })

        println(s"${warningMessages.size} Warnings where found.")
      }

      val result = typeCheckResult.getResult()
      val statusCode = result.astNode match {
        case _: DocumentNode =>
          runMapping(config, parsingContext, typeCheckResult)
        case _: ModuleNode =>
          runModule(config, parsingContext, typeCheckResult)
      }
      System.exit(statusCode)
    } else {

      println(typeCheckResult.messages().errorMessageString())

      println(typeCheckResult.messages().warningMessageString())

      if (typeCheckResult.hasErrors() || typeCheckResult.isEmpty()) {
        println(red(s"${typeCheckResult.errorMessages().size} Errors where found."))
        System.exit(1)
        return
      }
    }

  }

  def usage(): Unit = {
    println("[-input <name> <value>]* [-scenario <path>] [-param | -p <paramValue>]* [-debug]? [-output <path>]? <weave file path>")
  }

  private def runModule(config: RunnerConfiguration, parsingContext: ParsingContext, typeCheckResult: PhaseResult[TypeCheckingResult[_ <: AstNode]]): Int = {
    implicit val ctx: EvaluationContext = EvaluationContext()
    val values = config.parameters.zipWithIndex.map((valueIndex) => (valueIndex._2.toString, StringValue(valueIndex._1))).toMap
    val value = typeCheckResult.asInstanceOf[PhaseResult[TypeCheckingResult[ModuleNode]]].getResult()
    val result = WeaveCompiler.runtimeModuleCompilation(value, parsingContext, new DefaultRuntimeModuleNodeCompiler())
    val engine = result.getResult().executable
    var responseCode: Int = 0
    if (config.debug) {
      val latch: CountDownLatch = new CountDownLatch(1)
      val protocol: TcpServerProtocol = TcpServerProtocol()

      val debuggingSession: DefaultWeaveDebuggingSession = engine.asInstanceOf[InterpretedModuleExecutableWeave].debug(protocol)
      debuggingSession.addSessionListener(() => {
        new Thread() {
          override def run(): Unit = {
            try {
              val value = engine.execute(values = values)
              if (NumberType.accepts(value)) {
                responseCode = NumberType.coerce(value).evaluate.toInt
              }
              protocol.disconnect()
              latch.countDown()
            } finally {
              ctx.close()
            }
          }
        }.start()
      })
      println("[dw-debugger] Waiting for debugger client to connect.")
      latch.await()
    } else {
      try {
        val value = engine.execute(values = values)
        if (NumberType.accepts(value)) {
          responseCode = NumberType.coerce(value).evaluate.toInt
        }
      } finally {
        ctx.close()
      }
    }

    responseCode
  }

  private def runMapping(config: RunnerConfiguration, parsingContext: ParsingContext, typeCheckResult: PhaseResult[TypeCheckingResult[_ <: AstNode]]): Int = {
    implicit val ctx: EvaluationContext = EvaluationContext()
    val value = typeCheckResult.asInstanceOf[PhaseResult[TypeCheckingResult[DocumentNode]]].getResult()
    val result = WeaveCompiler.runtimeCompilation(value, parsingContext, new DefaultRuntimeModuleNodeCompiler())
    val engine = result.getResult().executable
    if (config.debug) {
      val latch: CountDownLatch = new CountDownLatch(1)
      val protocol: TcpServerProtocol = TcpServerProtocol()
      val debuggingSession: DefaultWeaveDebuggingSession = engine.asInstanceOf[InterpretedMappingExecutableWeave].debug(protocol)
      debuggingSession.addSessionListener(() => {
        new Thread() {
          override def run(): Unit = {
            val values: Map[String, Value[_]] = contextValues(config)
            engine.writeWith(buildWriter(engine, config.output, Some(new WeaveDataFormat())), buildReaders(engine, config.inputs.toMap), values)
            protocol.disconnect()
            latch.countDown()
          }
        }.start()
      })
      println("[dw-debugger] Waiting for debugger client to connect.")
      latch.await()
    } else {
      val values: Map[String, Value[_]] = contextValues(config)
      engine.writeWith(buildWriter(engine, config.output, Some(new WeaveDataFormat())), buildReaders(engine, config.inputs.toMap), values)
    }

    0 //OK
  }

  private def contextValues(config: RunnerConfiguration): Map[String, Value[_]] = {
    val values = config.scenarioDir
      .map((folder) => {
        val elements = InputElementFactory.fromDirectory(new File(folder, "inputs").listFiles())
        WeaveRuntimeUtils.toContext(elements)
      })
      .getOrElse(Map[String, Value[_]]())
    values
  }
}

case class RunnerConfiguration(resource: WeaveResource, nameIdentifier: NameIdentifier, scenarioDir: Option[File], output: OutputStream = System.out, inputs: Seq[(String, File)] = Seq(), parameters: Seq[String] = Seq(), debug: Boolean = false)
