package org.mule.weave.v2.interpreted

import org.mule.weave.v2.codegen.CodeGenerator
import org.mule.weave.v2.codegen.StringCodeWriter
import org.mule.weave.v2.interpreted.module.WeaveWriter
import org.mule.weave.v2.interpreted.module.WeaveWriterSettings
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.option.ConfigurableSchemaSetting
import org.mule.weave.v2.module.option.SchemaStreamAwareSetting
import org.mule.weave.v2.module.writer.EmptyWriter
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.ast.structure.{ DocumentNode => AstDocumentNode }

import java.io.File
import java.io.FileOutputStream
import java.io.PrintWriter
import java.io.StringWriter
import java.time.Instant
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.util.logging.Level
import java.util.logging.Logger
import scala.util.Try

class Dumper(values: Map[String, Value[_]], e: Exception, writer: Writer = EmptyWriter, astDocument: AstDocumentNode)(implicit ctx: EvaluationContext) {

  def dumpGlobalVars(dumpFolder: File): Unit = {
    values.foreach(nameValuePair => {
      nameValuePair._2 match {
        case dc: DumpCapable => dc.dumpValue(dumpFolder)
        case _ => {
          Dumper.dumpValue(s"${nameValuePair._1}.dwl", (dumpFile: File) => Dumper.dumpUsingWeave(dumpFile, nameValuePair._2), "Global Var", dumpFolder)
        }
      }
    })
  }

  def dumpScript(dumpFolder: File): Unit = {
    val dumpScriptFile = new File(dumpFolder, "script.dwl")
    val pw = new PrintWriter(dumpScriptFile)
    val writer: StringCodeWriter = new StringCodeWriter()
    val codeGenerator: CodeGenerator = new CodeGenerator(writer)
    try {
      codeGenerator.generate(astDocument)
      pw.write(writer.codeContent())
    } catch {
      case e: Exception => pw.write(s"Error generating the DW script from the astDocumentNode: $e")
    } finally {
      pw.close()
    }
  }

  def dumpException(dumpFolder: File): Unit = {
    val dumpExceptionFile = new File(dumpFolder, "exception.txt")
    val pw = new PrintWriter(dumpExceptionFile)
    try {
      e.printStackTrace(pw)
    } finally {
      pw.close()
    }
  }

  def dumpFilesToReproduceError(): Unit = {
    val fmt = DateTimeFormatter.ofPattern("yyyy-MM-dd'_'HH.mm.ss.SSS").withZone(ZoneId.systemDefault())
    val instantFolder = fmt.format(Instant.now())
    val baseFolderPath = ctx.serviceManager.settingsService.dumper().workingDirectory()
    val dumpBaseFolder = new File(baseFolderPath, Dumper.DUMP_FOLDER)
    val dumpFolder = new File(dumpBaseFolder, instantFolder)
    dumpFolder.mkdirs()
    dumpScript(dumpFolder)
    dumpGlobalVars(dumpFolder)
    dumpException(dumpFolder)
    writer.settings match {
      case configurableSchema: ConfigurableSchemaSetting => configurableSchema.dumpSchema(dumpFolder, "writerSchema")
      case flatFileSchema: SchemaStreamAwareSetting => flatFileSchema.dumpFlatFileSchema(dumpFolder, ctx, "schemaFlatFileWriter.ffd")
      case _ =>
    }
    ctx.serviceManager.loggingService.logInfo(s"\nThe files to reproduce the DataWeave error are in ${dumpFolder.getCanonicalPath}")
  }

}

object Dumper {

  val DUMP_FOLDER = "dw-dump"

  val logger: Logger = Logger.getLogger(getClass.getName)

  def apply(values: Map[String, Value[_]], e: Exception, writer: Writer = EmptyWriter, executableDocument: DocumentNode, astDocument: AstDocumentNode)(implicit ctx: EvaluationContext): Dumper = {
    new Dumper(values, e, writer, astDocument)
  }

  def dumpUsingWeave(dumpFile: File, value: Value[_])(implicit ctx: EvaluationContext): Unit = {
    val settings = new WeaveWriterSettings()
    settings.set("indent", " ")
    val outputStream: FileOutputStream = new FileOutputStream(dumpFile)
    val we: WeaveWriter = WeaveWriter(outputStream, settings)
    try {
      we.writeValue(value)
    } catch {
      case e: Exception => {
        dumpFile.delete()
        logger.log(Level.WARNING, s"The value ${dumpFile.getName} cannot be dumped.")
        if (ctx.serviceManager.settingsService.dumper().fillStackTrace) {
          val stringWriter = new StringWriter()
          e.printStackTrace(new PrintWriter(stringWriter))
          logger.log(Level.WARNING, s"DataWeave Dumper issue stacktrace: \n${stringWriter.toString}")
        }
      }
    } finally {
      Try(we.close())
      Try(outputStream.close())
    }
  }

  def dumpValue(fileName: String, wayToDump: File => Unit, originOfValue: String, dumpFolder: File)(implicit ctx: EvaluationContext): Unit = {
    val dumpFile = new File(dumpFolder, fileName)
    try {
      wayToDump(dumpFile)
    } catch {
      case e: Exception => {
        dumpFile.delete()
        logger.log(Level.WARNING, s"\nThe $originOfValue $fileName file cannot be dumped.")
        if (ctx.serviceManager.settingsService.dumper().fillStackTrace) {
          val stringWriter = new StringWriter()
          e.printStackTrace(new PrintWriter(stringWriter))
          logger.log(Level.WARNING, s"DataWeave Dumper issue stacktrace: \n${stringWriter.toString}")
        }
      }
    }
  }

}
