package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.node.ExecutionNode
import org.mule.weave.v2.interpreted.node.ModuleNode
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.exception.InternalErrorException
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.utils.SchemaNodeOrigin

import scala.collection.mutable.ArrayBuffer

class EngineGrammarTransformation(val parsingContext: ParsingContext, val scopeNavigator: ScopesNavigator, val moduleLoader: RuntimeModuleNodeCompiler)
    extends EngineDirectiveTransformations
    with EngineDocumentTransformations
    with EngineFunctionTransformations
    with EngineHeaderTransformations
    with EnginePatternTransformations
    with EngineSchemaTransformations
    with EngineStructureTransformations
    with EngineVariableTransformations
    with EngineModuleTransformations
    with EngineUndefinedTransformations
    with EngineUpdaterTransformations {

  var rootElement: AstNode = _

  override val _variablesTable: VariableTable = new VariableTable()

  override val _modulesNameTable: VariableTable = new VariableTable()

  override val _externalBindings: ExternalBindings = new ExternalBindings()

  def transformDocument(document: parser.ast.structure.DocumentNode): DocumentNode = {
    rootElement = document
    try {
      transform(document)
    } finally {
      rootElement = null
    }
  }

  def transformModule(module: parser.ast.module.ModuleNode): ModuleNode = {
    rootElement = module
    try {
      transform(module)
    } finally {
      rootElement = null
    }
  }

  def transform[T <: ExecutionNode](astNode: parser.ast.AstNode): T = {
    astNode match {
      case baseAstNode: AstNode => {
        implicit val location: Location = astNode.location()

        val result = baseAstNode match {
          //
          case dn: parser.ast.structure.DocumentNode => transformDocumentNode(dn)
          case mn: parser.ast.module.ModuleNode => transformModuleNode(mn)
          //
          case parser.ast.header.HeaderNode(directives) => transformHeaderNode(directives)
          case parser.ast.UndefinedExpressionNode() => transformUndefinedNode()
          //
          case parser.ast.header.directives.NamespaceDirective(prefix, uri, _) => transformNamespaceDirective(prefix, uri)
          case parser.ast.header.directives.VersionMajor(v) => transformVersionMajor(v)
          case parser.ast.header.directives.DirectiveOption(name, value) => transformDirectiveOption(name, value)
          case parser.ast.header.directives.TypeDirective(variable, params, t, codeAnnotations) => transformTypeDirective(variable, params, t, codeAnnotations)
          case parser.ast.header.directives.VersionDirective(major, minor, _) => transformVersionDirective(major, minor)
          case parser.ast.header.directives.OutputDirective(id, mime, options, _, _) => transformOutputDirective(id, mime, options)
          case parser.ast.header.directives.VersionMinor(v) => transformVersionMinor(v)
          case parser.ast.header.directives.VarDirective(variable, value, _, annotations) => transformVarDirective(variable, value, annotations)
          case fn @ parser.ast.header.directives.FunctionDirectiveNode(variable, literal, codeAnnotations) => transformFunctionDirective(variable, literal, codeAnnotations, fn)
          //
          case parser.ast.patterns.PatternMatcherNode(lhs, patterns, _) => transformPatternMatcherNode(lhs, patterns)

          case parser.ast.patterns.ExpressionPatternNode(pattern, name, function) => transformExpressionPatternNode(pattern, name, function)
          case parser.ast.patterns.LiteralPatternNode(pattern, name, function) => transformLiteralPatternNode(pattern, name, function)
          case parser.ast.patterns.RegexPatternNode(pattern, name, function) => transformRegexPatternNode(pattern, name, function)
          case parser.ast.patterns.DefaultPatternNode(value, name) => transformDefaultPatternNode(name, value)
          case parser.ast.patterns.TypePatternNode(pattern, name, function) => transformTypePatternNode(pattern.asInstanceOf[WeaveTypeNode], name, function)
          case parser.ast.patterns.EmptyArrayPatternNode(function) => transformEmptyArrayNode(function)
          case parser.ast.patterns.DeconstructArrayPatternNode(head, tail, function) => transformDeconstructArrayNode(head, tail, function)

          case parser.ast.patterns.EmptyObjectPatternNode(function) => transformEmptyObjectNode(function)
          case parser.ast.patterns.DeconstructObjectPatternNode(headKey, headValue, tail, function) => transformDeconstructObjectNode(headKey, headValue, tail, function)
          //
          case parser.ast.structure.DateTimeNode(v, _) => transformDateTimeNode(v)
          case sn: parser.ast.structure.StringNode => transformStringNode(sn)

          case parser.ast.structure.TimeNode(v, _) => transformTimeNode(v)
          case parser.ast.structure.StringInterpolationNode(v) => transformStringInterpolationNode(v)
          case parser.ast.structure.LocalDateTimeNode(v, _) => transformLocalDateTimeNode(v)
          case un @ parser.ast.structure.UriNode(v, _) => transformUriNode(v, un)
          case parser.ast.structure.ConditionalNode(value, cond) => transformConditionalNode(value, cond)
          case parser.ast.structure.PeriodNode(v, _) => transformPeriodNode(v)
          case parser.ast.structure.RegexNode(regex, _) => transformRegexNode(regex)
          case trn: parser.ast.types.TypeReferenceNode => transformTypeReferenceNode(trn)
          case wType: parser.ast.types.WeaveTypeNode => transformTypeNode(wType)
          case parser.ast.structure.TimeZoneNode(v, _) => transformTimeZoneNode(v)
          case parser.ast.structure.LocalDateNode(v, _) => transformLocalDateNode(v)
          case parser.ast.structure.NullNode(_) => transformNullNode()
          case an: parser.ast.structure.ArrayNode => transformArrayNode(an)
          case parser.ast.structure.HeadTailArrayNode(head, tail, _) => transformHeadTailArrayNode(head, tail)
          case parser.ast.structure.HeadTailObjectNode(headKey, headValue, tail, _) => transformHeadTailObjectNode(headKey, headValue, tail)

          case parser.ast.structure.AttributesNode(attrs) => transformAttributesNode(attrs)
          case on: parser.ast.structure.ObjectNode => transformObjectNode(on)
          case parser.ast.functions.FunctionNode(params, body, returnType, typeParameterList) => transformFunctionNode(params.paramList, body, returnType, typeParameterList)
          case parser.ast.structure.NameValuePairNode(key, value, cond) => transformNameValuePairNode(key, value, cond)
          case parser.ast.structure.NamespaceNode(prefix) => transformNamespaceNode(prefix)
          case parser.ast.functions.FunctionParameter(variable, defaultValue, wtype, codeAnnotations) => transformFunctionParameter(variable, defaultValue, wtype, codeAnnotations)
          case parser.ast.structure.LocalTimeNode(v, _) => transformLocalTimeNode(v)
          case parser.ast.structure.KeyValuePairNode(key, value, cond) => transformKeyValuePairNode(key, value, cond)
          case parser.ast.structure.NameNode(keyName, cond, _) => transformNameNode(keyName, cond)
          case parser.ast.structure.DynamicNameNode(keyName) => transformDynamicNameNode(keyName)
          case parser.ast.structure.NumberNode(v, _) => transformNumberNode(baseAstNode.location(), v)
          case parser.ast.structure.BooleanNode(v, _) => transformBooleanNode(v)
          case kn: parser.ast.structure.KeyNode => transformKeyNode(kn)
          case parser.ast.structure.DynamicKeyNode(keyName, attr) => transformDynamicKeyNode(keyName, attr)
          case parser.ast.functions.UsingNode(vars, expr, _) => transformUsing(vars.assignmentSeq, expr)
          case doBlock: parser.ast.functions.DoBlockNode => transformDoBlock(doBlock)
          case parser.ast.functions.UsingVariableAssignment(name, expr) => transformVarDirective(name, expr, Seq())
          //
          case parser.ast.selectors.NullSafeNode(selector, _) => transformNullSafeNode(selector)
          case parser.ast.selectors.NullUnSafeNode(selector, _) => transformNullUnSafeNode(selector)
          case parser.ast.selectors.ExistsSelectorNode(selectable) => transformExistsSelectorNode(selectable)
          //
          case vrn: parser.ast.variables.VariableReferenceNode => transformVariableReferenceNode(vrn)
          case ni: parser.ast.variables.NameIdentifier => transformNameSlot(ni)
          //
          case parser.ast.conditional.UnlessNode(ifExpr, condition, elseExpr, _) => transformUnlessNode(ifExpr, condition, elseExpr)
          case parser.ast.conditional.IfNode(expression, condition, elseExpr, _) => transformIfNode(expression, condition, elseExpr)
          case parser.ast.conditional.DefaultNode(lhs, rhs, _) => transformDefaultNode(lhs, rhs)
          //
          case node @ parser.ast.functions.FunctionCallNode(function, args, typeParams, _) => transformFunctionCallNode(function, args.args, node, typeParams)
          case node @ parser.ast.operators.BinaryOpNode(name, lhs, rhs, _) => transformBinaryOpNode(node, name, lhs, rhs)
          case node @ parser.ast.operators.UnaryOpNode(name, rhs, _) => transformUnaryOpNode(node, name, rhs)
          //
          case parser.ast.structure.schema.SchemaPropertyNode(name, value, condition) => transformSchemaPropertyNode(name, value, condition)
          case node @ parser.ast.structure.schema.SchemaNode(properties) => transformSchemaNode(properties, SchemaNodeOrigin(node))
          //
          //
          case parser.ast.logical.AndNode(lhs, rhs, _) => transformAndNode(lhs, rhs)
          case parser.ast.logical.OrNode(lhs, rhs, _) => transformOrNode(lhs, rhs)
          //
          // UPDATES!!!!
          case un: parser.ast.updates.UpdateNode => transformUpdate(un)
          case un: parser.ast.updates.UpdateExpressionsNode => transformUpdateExpressionsNode(un)
          case un: parser.ast.updates.UpdateExpressionNode => transformUpdateExpressionNode(un)
          case un: parser.ast.updates.UpdateSelectorNode => transformUpdateSelectorNode(un)
          case _ => throw new InternalErrorException(astNode.location(), s"No conversion available for ${astNode.getClass}")
        }

        result
        val resultAstNode: ExecutionNode = result.asInstanceOf[ExecutionNode]

        resultAstNode._location = Some(astNode.location())
        resultAstNode.asInstanceOf[T]
      }
      case _ => throw new InternalErrorException(astNode.location(), s"No conversion available for ${astNode.getClass}")
    }
  }

  override val _modules: ArrayBuffer[VariableTable] = ArrayBuffer()

}

object EngineGrammarTransformation {

  def apply(parsingContext: ParsingContext, scopeNavigator: ScopesNavigator, moduleNodeLoader: RuntimeModuleNodeCompiler): EngineGrammarTransformation = {
    new EngineGrammarTransformation(parsingContext, scopeNavigator, moduleNodeLoader)
  }
}
