package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.wrappers.ContextualizedValue

import scala.collection.mutable

object ArrayDistinctFunctionValue extends BinaryFunctionValue {
  val L = ArrayType

  val R = FunctionType

  override def doExecute(leftValue: L.V, fn: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val distinctIterator = new ArrayDistinctIterator(leftValue.evaluate.toIterator(), fn)
    ArrayValue(ArraySeq(distinctIterator, materializedValues = true), this)
  }
}

object ObjectDistinctFunctionValue extends BinaryFunctionValue {
  val L = ObjectType
  val R = FunctionType

  override def doExecute(leftValue: L.V, fn: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val seq = ObjectSeq(new ObjectDistinctIterator(leftValue.evaluate.toIterator(), fn), materializedValues = true)
    ObjectValue(seq)
  }
}

class ArrayDistinctIterator(iterator: Iterator[Value[_]], fn: FunctionValue)(implicit ctx: EvaluationContext) extends Iterator[Value[_]] {

  var nextValue: Value[_] = _
  var index = 0

  private val uniques = mutable.HashSet[ContextualizedValue]()

  def calculateNextValue(): Unit = {
    while (nextValue == null && iterator.hasNext) {

      val materializedValue: Value[_] = iterator.next().materialize
      val criteriaValue: Value[_] = fn.call(materializedValue, NumberValue(index))
      index = index + 1
      //Need to materialize the value as is needed for comparison over and over
      val contextValue: ContextualizedValue = ContextualizedValue(criteriaValue.materialize)

      if (!uniques.contains(contextValue)) {
        nextValue = materializedValue
        //Add it to the uniques as is a new one
        uniques.add(contextValue)
      }
    }
  }

  override def hasNext: Boolean = {
    if (nextValue == null) {
      calculateNextValue()
    }
    nextValue != null
  }

  override def next(): Value[_] = {
    if (hasNext) {
      val value = nextValue
      nextValue = null
      value
    } else {
      Iterator.empty.next()
    }
  }
}

class ObjectDistinctIterator(iterator: Iterator[KeyValuePair], fn: FunctionValue)(implicit ctx: EvaluationContext) extends Iterator[KeyValuePair] {

  var nextValue: KeyValuePair = _

  private val uniques = mutable.HashSet[ContextualizedValue]()

  def calculateNextValue(): Unit = {
    while (nextValue == null && iterator.hasNext) {

      val materializedValue = iterator.next().materialize
      val criteriaValue = fn.call(materializedValue._2, materializedValue._1)
      val contextValue = ContextualizedValue(criteriaValue.materialize)

      if (!uniques.contains(contextValue)) {
        nextValue = materializedValue
        //Add it to the uniques as is a new one
        uniques.add(contextValue)
      }
    }
  }

  override def hasNext: Boolean = {
    if (nextValue == null) {
      calculateNextValue()
    }
    nextValue != null
  }

  override def next(): KeyValuePair = {
    if (hasNext) {
      val value = nextValue
      nextValue = null
      value
    } else {
      Iterator.empty.next()
    }
  }
}
