/**
* This module contains helper functions to work with Objects.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Objects` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Returns a list of key-value pairs that describe the key, value, and any
 * attributes in the input object.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `obj` | The `Object` to describe.
 * |===
 *
 * == Example
 *
 * This example returns the key, value, and attributes in the object specified
 * in the variable `myVar`.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Objects
 * var myVar = read('<xml attr="x"><a>true</a><b>1</b></xml>', 'application/xml')
 * output application/json
 * ---
 * { "entrySet" : entrySet(myVar) }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "entrySet": [
 *     {
 *        "key": "xml",
 *        "value": {
 *          "a": "true",
 *          "b": "1"
 *        },
 *        "attributes": {
 *          "attr": "x"
 *        }
 *     }
 *   ]
 * }
 * ----
 */
fun entrySet<T <: Object>(obj: T) =
  obj pluck (value, key) -> {
    key: key,
    value: value,
    attributes: key.@
  }

/**
* Returns a list of keys from an object.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `obj` | The object to evaluate.
* |===
*
* == Example
*
* This example returns the keys from the input object.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Objects
* output application/json
* ---
* { "nameSet" : nameSet({ "a" : true, "b" : 1}) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "nameSet" : ["a","b"] }
* ----
*/
fun nameSet(obj: Object): Array<String> = obj pluck ($$ as String)

/**
* Returns the list of key names from an object.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `object` | The object to evaluate.
* |===
*
* == Example
*
* This example returns the keys from the input object.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Objects
* output application/json
* ---
* { "keySet" : keySet({ "a" : true, "b" : 1}) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "keySet" : ["a","b"] }
* ----
*
* == Example
*
* This example illustrates a difference between `keySet` and `nameSet`.
* Notice that `keySet` retains the attributes (`name` and `lastName`)
* and namespaces (`xmlns`) from the XML input, while `nameSet` returns
* `null` for them because it does not retain them.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Objects
* var myVar = read('<users xmlns="http://test.com">
*                      <user name="Mariano" lastName="Achaval"/>
*                      <user name="Stacey" lastName="Duke"/>
*                   </users>', 'application/xml')
* output application/json
* ---
* { keySetExample: flatten([keySet(myVar.users) map $.#,
*                           keySet(myVar.users) map $.@])
* }
* ++
* { nameSet: flatten([nameSet(myVar.users) map $.#,
*                     nameSet(myVar.users) map $.@])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "keySet": [
*     "http://test.com",
*     "http://test.com",
*     {
*       "name": "Mariano",
*       "lastName": "Achaval"
*     },
*     {
*       "name": "Stacey",
*       "lastName": "Duke"
*     }
*   ],
*   "nameSet": [
*     null,
*     null,
*     null,
*     null
*   ]
* }
* ----
*/
fun keySet<T <: Object>(obj: T): ? = obj pluck $$

/**
* Returns the list of the values in an object.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `obj` | The object to evaluate.
* |===
*
* == Example
*
* This example returns the values from the input object.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import dw::core::Objects
* output application/json
* ---
* { "valueSet" : valueSet({a: true, b: 1}) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "valueSet" : [true,1] }
* ----
*/
fun valueSet <K,V>(obj: {(K)?: V}): Array<V> = obj pluck $

/**
 * Appends any key-value pairs from a source object to a target object.
 *
 *
 * If source and target objects have the same key, the function appends
 * that source object to the target and removes that target object from the output.
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `source` | The object to append to the `target`.
 * | `target` | The object to which the `source` object is appended.
 * |===
 *
 * == Example
 *
 * This example appends the source objects to the target. Notice that
 * `"a" : true,` is removed from the output, and `"a" : false` is appended
 * to the target.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import mergeWith from dw::core::Objects
 * output application/json
 * ---
 * { "mergeWith" : { "a" : true, "b" : 1} mergeWith { "a" : false, "c" : "Test"} }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * "mergeWith": {
 *     "b": 1,
 *     "a": false,
 *     "c": "Test"
 * }
 * ----
 */
fun mergeWith<T <: Object,V <: Object>(source: T, target: V): ? =
  (source -- keySet(target)) ++ target

// Helper function. Do not publish doc.
/**
* Helper method to make `mergeWith` null friendly.
*/
fun mergeWith<T <: Object>(a: Null, b: T): T = b

// Helper function. Do not publish doc.
/**
* Helper method to make `mergeWith` null friendly.
*/
fun mergeWith<T <: Object>(a: T, b: Null): T = a

/**
* Breaks up an object into sub-objects that contain the specified number of
* key-value pairs.
*
*
* If there are fewer key-value pairs in an object than the specified number, the
* function will fill the object with those pairs. If there are more pairs, the
* function will fill another object with the extra pairs.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `items` | Key-value pairs in the source object.
* | `amount` | The number of key-value pairs allowed in an object.
* |===
*
* == Example
*
* This example breaks up objects into sub-objects based on the specified `amount`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import divideBy from dw::core::Objects
* output application/json
* ---
* { "divideBy" : {"a": 1, "b" : true, "a" : 2, "b" : false, "c" : 3} divideBy 2 }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "divideBy": [
*     {
*       "a": 1,
*       "b": true
*     },
*     {
*       "a": 2,
*       "b": false
*     },
*     {
*       "c": 3
*     }
*   ]
* }
* ----
*/
fun divideBy(items: Object, amount: Number): Array<{}> = do {
    fun internalDivideBy<T>(items: Object, amount: Number, carry:{} ): Array<{}> =
        items match {
          case {k:v ~ xs} ->
            if(sizeOf(carry) == amount - 1)
                [carry ++ {(k):v} ~ internalDivideBy(xs, amount, {})]
            else
               internalDivideBy(xs, amount, carry ++ {(k):v} )
          else ->
            if(isEmpty(carry))
             []
            else
             [carry]
        }
    ---
    internalDivideBy(items, amount, {})
}
