/**
 * This module contains core DataWeave functions for data transformations.
 * It is automatically imported into any DataWeave script.
 */
%dw 2.0

/**
 * These are the native types of DataWeave.
 *
 * They are the only types that allow the `???` definition.
 */

/**
 * `String` type
 */
type String = ???
/**
* A `Boolean` type of `true` or `false`.
*/
type Boolean = ???
/**
* A number type: Any number, decimal, or integer is represented by the Number` type.
*/
type Number = ???
/**
* A `Range` type represents a sequence of numbers.
*/
type Range = ???
/**
* A `Namespace` type represented by a `URI` and a prefix.
*/
type Namespace = ???
/**
* A Uri.
*/
type Uri = ???
/**
* A `Date` and `Time` within a `TimeZone`. For example: `|2018-09-17T22:13:00Z|`
*/
type DateTime = ???
/**
* A `DateTime` in the current `TimeZone`. For example: `|2018-09-17T22:13:00|`
*/
type LocalDateTime = ???
/**
* A date represented by a year, month, and day. For example: `|2018-09-17|`
*/
type Date = ???
/**
* A `Time` in the current `TimeZone`. For example: `|22:10:18|`
*/
type LocalTime = ???
/**
* A time in a specific `TimeZone`. For example: `|22:10:18Z|`
*/
type Time = ???
/**
* A time zone.
*/
type TimeZone = ???
/**
* A period.
*/
type Period = ???
/**
* A blob.
*/
type Binary = ???
/**
* A Null type.
*/
type Null = ???
/**
* A regex type.
*/
type Regex = ???
/**
* Bottom type. This type can be assigned to all the types.
*/
type Nothing = ???
/**
 * The top-level type. `Any` extends all of the system types, which
 * means that anything can be assigned to a `Any` typed variable.
 */
type Any = ???
/**
 * Array type that requires a `Type(T)` to represent the elements of the list.
 * Example: `Array<Number>` represents an array of numbers, and `Array<Any>`
 * represents an array of any type.
 */
type Array<T> = ???
/**
 * Type that represents any object, which is a collection of `Key` and value pairs.
 */
type Object = ???
/**
* Represents a type in the DataWeave type system.
*/
type Type<T> = ???
/**
* A key of an `Object`.
*/
type Key = ???

/**
* Generic dictionary interface.
*/
type Dictionary<T> = {_?: T}

/**
* A union type that represents all the types that can be compared to each other.
*/
type Comparable = String | Number | Boolean | DateTime | LocalDateTime | Date | LocalTime | Time | TimeZone

/**
* A union type that represents all the simple types.
*/
type SimpleType = String | Boolean | Number | DateTime | LocalDateTime | Date | LocalTime | Time | TimeZone | Period

/**
* Logs the specified value with an optional `prefix`, then returns the
* value unchanged. The function logs the output as a system log.
*
* This function can be used to debug DataWeave scripts until there a proper
* debugger is incorporated.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `prefix` | A string that typically describes the log.
* | `value` | The value to log.
* |===
*
* == Example
*
* This example produces the output shown below in a Logger (through the Mule
* `LoggerMessageProcessor`), while the Mule `DefaultLoggingService` prints
* `My Age - 33` in the console output.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/xml
* var myvar = { "age" : 33 }
* ---
* { "age": log("My Age", myvar.age) }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <age>33</age>
* ----
*/
fun log <T>(prefix: String = "", value: T): T = native("system::log")

/**
* Reads the input string or binary and returns parsed
* content.
*
*
* This function can be useful if the reader cannot determine the content type
* by default.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `stringToParse` | The string or binary to read.
* | `contentType` | A supported format (or content type). Default: `application/dw`.
* | `readerProperties` | Optional: Sets reader configuration properties. For other formats and reader configuration properties, see https://docs.mulesoft.com/mule-runtime/4.1/dataweave-formats[Formats Supported by DataWeave].
* |===
*
* == Example
*
* This example reads a string as a CSV format without a header and outputs it
* to JSON. Notice that is adds column names as keys to the output object. Also,
* if you do not append `[0]` to the function call, the results will return as an
* array (with square brackets surrounding the entire output object).
*
* === Source
*
* ----
* %dw 2.0
* var myVar = "Some, Body"
* output application/json
* ---
* read(myVar,"application/csv",{header:false})[0]
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "column_0": "Some",
*   "column_1": " Body"
* }
* ----
*
* == Example
*
* This example reads the input XML and shows the syntax for a reader property.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* {
*    "XML" : read("<prices><basic>9.99</basic></prices>",
*                 "application/xml",
*                 { indexedReader: "false" })."prices"
* }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version='1.0' encoding='UTF-8'?>
* <XML>
*   <basic>9.99</basic>
*   <premium>53</premium>
*   <vip>398.99</vip>
* </XML>
* ----
*/
fun read(stringToParse: String | Binary, contentType: String = "application/dw", readerProperties: Object = {}) = native("system::read")

/**
* Similar to the `read` function. However, `readURL` accepts a URL as input.
* Otherwise, it accepts the same arguments as `read`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `url` | The URL string to read.
* | `contentType` | A supported format (or content type). Default: `application/dw`.
* | `readerProperties` | Optional: Sets reader configuration properties. For other formats and reader configuration properties, see https://docs.mulesoft.com/mule-runtime/4.1/dataweave-formats[Formats Supported by DataWeave].
* |===
*
* == Example
*
* This example reads JSON object from a URL. (For readability, the output values
* shown below are shortened (`...`).)
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* readUrl("https://jsonplaceholder.typicode.com/posts/1", "application/json")
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "userId": 1,
*   "id": 1,
*   "title": "sunt aut facere ...",
*   "body": "quia et suscipit\nsuscipit ..."
* }
* ----
*/
fun readUrl(url: String, contentType: String = "application/dw", readerProperties: Object = {}) = native("system::readUrl")

/**
* Writes (as a String or Binary) the given value in a specific format.
*
*
* Returns a string or binary with the serialized representation of the value in the specified mimeType (format).
* You might use this function when you need some data in a format different from the script output, for example to embed data as JSON/CSV inside an XML.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `value` | The value to write. The value can be of any supported data type.
* | `contentType` | A supported format (or content type) to write. Default: `application/dw`.
* | `writerProperties` | Optional: Sets writer configuration properties. For writer configuration properties (and other content types), see https://docs.mulesoft.com/mule-runtime/4.1/dataweave-formats[Formats Supported by DataWeave].
* |===
*
* == Example
*
* This example takes JSON input and writes the payload to a CSV format that uses a
* pipe (`|`) separator and includes the header (matching keys in the JSON objects).
* Note that if you instead use `"header":false` in your script, the output will
* lack the `Name|Email|Id|Title` header in the output.
*
* === Input
*
* [source,JSON,linenums]
* ----
* [
*   {
*     "Name": "Mr White",
*     "Email": "white@mulesoft.com",
*     "Id": "1234",
*     "Title": "Chief Java Prophet"
*   },
*   {
*     "Name": "Mr Orange",
*     "Email": "orange@mulesoft.com",
*     "Id": "4567",
*     "Title": "Integration Ninja"
*   }
* ]
* ----
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* {
*   "output" : write(payload, "application/csv", {"header":true, "separator" : "|"})
* }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="US-ASCII"?>
* <output>Name|Email|Id|Title
* Mr White|white@mulesoft.com|1234|Chief Java Prophet
* Mr Orange|orange@mulesoft.com|4567|Integration Ninja
* </output>
* ----
*/
fun write (value: Any, contentType: String = "application/dw", writerProperties: Object = {}): String | Binary = native("system::write")

/**
* Returns a pseudo-random number greater than or equal to 0.0 and less than 1.0.
*
*
* == Example
*
* This example generates a pseudo-random number and multiplies it by 1000.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { price: random() * 1000 }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "price": 65.02770292248383 }
* ----
*/
fun random(): Number = native("system::random")

/**
* Returns a pseudo-random whole number from 0 to the specified number
* (exclusive).
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `upperBound` | A number that sets the upper bound of the random number.
* |===
*
* == Example
*
* This example returns an integer from 0 to 1000 (exclusive).
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { price: randomInt(1000) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "price": 442.0 }
* ----
*/
fun randomInt(upperBound: Number): Number = floor(random() * upperBound)

/**
* Returns a v4 UUID using random numbers as the source.
*
*
* == Example
*
* This example generates a random v4 UUID.
*
* === Source
*
* ----
* %dw 2.0
* output application/json
* ---
* uuid()
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* "7cc64d24-f2ad-4d43-8893-fa24a0789a99"
* ----
*/
fun uuid(): String = native("system::uuid")


/**
* Returns a `DateTime` object with the current date and time.
*
*
* == Example
*
* This example shows uses of the `now()` function with valid
* selectors. It also shows how to get the epoch time with `now() as Number`.
* See also,
* https://docs.mulesoft.com/mule-runtime/4.1/dataweave-selectors[DataWeave Selectors].
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   now: now(),
*   epochTime : now() as Number,
*   nanoseconds: now().nanoseconds,
*   milliseconds: now().milliseconds,
*   seconds: now().seconds,
*   minutes: now().minutes,
*   hour: now().hour,
*   day: now().day,
*   month: now().month,
*   year: now().year,
*   quarter: now().quarter,
*   dayOfWeek: now().dayOfWeek,
*   dayOfYear: now().dayOfYear,
*   offsetSeconds: now().offsetSeconds
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "now": "2018-07-23T22:03:04.829Z",
*   "epochTime": 1533143270,
*   "nanoseconds": 829000000,
*   "milliseconds": 829,
*   "seconds": 4,
*   "minutes": 3,
*   "hour": 22,
*   "day": 23,
*   "month": 7,
*   "year": 2018,
*   "quarter": 3,
*   "dayOfWeek": 1,
*   "dayOfYear": 204,
*   "offsetSeconds": 0
* }
* ----
*/
fun now(): DateTime = native("system::now")

// Internal. Do not publish doc.
/**
* Internal method used to indicate that a function implementation is not
* written in DataWeave but in Scala.
*/
fun native(identifier: String): Nothing = ??? //This function is just a place holder

// A type: Iterator
/**
* This type is based on the
* https://docs.oracle.com/javase/8/docs/api/java/util/Iterator.html[iterator Java class].
* The iterator contains a collection and includes methods to iterate through
* and filter it.
*
* Just like the Java class, `Iterator` is designed to be consumed only once. For
* example, if you pass it to a
* https://docs.mulesoft.com/mule-runtime/4.1/logger-component-reference[Logger component],
* the Logger consumes it, so it becomes unreadable by further elements in the flow.
**/
type Iterator = Array {iterator: true}

//A type: Enum
/**
* This type is based on the
* https://docs.oracle.com/javase/7/docs/api/java/lang/Enum.html[Enum Java class].
*
* It must always be used with the `class` property, specifying the full Java
* class name of the class, as shown in the example below.
*
* .Transform
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/java
* ---
* "Male" as Enum {class: "com.acme.GenderEnum"}
* ----
*/
type Enum = String {enumeration: true}

// A type: NaN.
/**
* `java.lang.Float` and `java.lang.Double` have special cases for `NaN` and `Infinit`.
* DataWeave does not have these concepts for its number multi-precision nature.
* So when it is mapped to DataWeave values, it is wrapped in a Null with a Schema marker.
*/
type NaN = Null {NaN: true}

// A type: CData.
/**
* XML defines a `CData` custom type that extends from `String` and is used
* to identify a CDATA XML block.
*
* It can be used to tell the writer to wrap the content inside CDATA or to
* check if the input string arrives inside a CDATA block. `:cdata` inherits
* from the type `:string`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* {
*   "users" :
*   {
*     "user" : "Mariano" as CData,
*     "age" : 31 as CData
*   }
* }
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <users>
*   <user><![CDATA[Mariano]]></user>
*   <age><![CDATA[31]]></age>
* </users>
* ----
**/
type CData = String {cdata: true}

//---------------------------------------------------------------------------------------------------------

/**
* Concatenates the elements of two lists (arrays) into a new list.
*
*
* If the two arrays contain different types of elements, the resulting array
* is all of `S` type elements of `Array<S>` followed by all the `T` type elements
* of `Array<T>`. Either of the arrays can also have mixed-type elements. Note
* that the arrays can contain any supported data type.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `source` | The source list (an `Array).
* | `with` | The list to concatenate with the source list.
* |===
*
* == Example
*
* The example concatenates an `Array<Number>` with an `Array<String>`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "result" : [0, 1, 2] ++ ["a", "b", "c"]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "result": [0, 1, 2, "a", "b", "c"]
* }
* ----
*
* == Example
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : [0, 1, true, "my string"] ++ [2, [3,4,5], {"a": 6}]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": [0, 1, true, "my string", 2, [3, 4, 5], { "a": 6}]
* }
* ----
**/
fun ++ <S,T>(source: Array<S> , with: Array<T>): Array<S | T> = native("system::ArrayAppendArrayFunctionValue")

/**
* Concatenates the characters of two strings.
*
*
* Strings are treated as arrays of characters, so the `++` operator concatenates
* the characters of each String as if they were arrays of single character
* String.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `source` | The source string.
* | `with` | The string to concatenate with the source string.
* |===
*
* == Example
*
* In the example, the `Mule` is treated as `Array<String> ["M", "u", "l", "e"]`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "name" : "Mule" ++ "Soft"
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "name": MuleSoft
* }
* ----
**/
fun ++(source: String, with: String): String = native("system::StringAppendStringFunctionValue")

//sduke: reformatted
/**
* Concatenates two objects and returns one flattened object.
*
*
* The `++` operator extracts all the key-values pairs from each object,
* then combines them together into one result object.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `source` | The source object.
* | `with` | The object to concatenate with the source object.
* |===
*
* == Example
*
* This example concatenates two objects and transforms them to XML.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* "concat" : {aa: "a", bb: "b"} ++ {cc: "c"}
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <concat>
*   <aa>a</aa>
*   <bb>b</bb>
*   <cc>c</cc>
* </concat>
* ----
**/
fun ++(source: Object , with: Object): Object = native("system::ObjectAppendObjectFunctionValue")


/**
* Appends a `LocalTime` with a `Date` to return a `LocalDateTime` value.
*
*
* `Date` and `LocalTime` instances are written in standard Java notation,
* surrounded by pipe (`|`) symbols. The result is a `LocalDateTime` object
* in the standard Java format. Note that the order in which the two objects are
* concatenated is irrelevant, so logically, `Date ++ LocalTime` produces the
* same result as `LocalTime ++ Date`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `date` | A `Date`.
* | `time` | A `LocalTime`, a time format without a time zone.
* |===
*
* == Example
*
* This example concatenates a `Date` and `LocalTime` object to return a
* `LocalDateTime`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "LocalDateTime" : (|2017-10-01| ++ |23:57:59|)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "LocalDateTime": "2017-10-01T23:57:59"
* }
* ----
**/
fun ++(date: Date , time: LocalTime): LocalDateTime = native("system::LocalDateAppendLocalTimeFunctionValue")

/**
* Appends a `LocalTime` with a `Date` to return a `LocalDateTime`.
*
*
* Note that the order in which the two objects are concatenated is irrelevant,
* so logically, `LocalTime ++ Date` produces the same result as
* `Date ++ LocalTime`.
*
* == Example
*
* This example concatenates `LocalTime` and `Date` objects to return a
* `LocalDateTime`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `time` | A `LocalTime`, a time format without a time zone.
* | `date` | A `Date`.
* |===
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "LocalDateTime" : (|23:57:59| ++ |2003-10-01|)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "LocalDateTime": "2017-10-01T23:57:59"
* }
* ----
**/
fun ++(time: LocalTime , date: Date): LocalDateTime = native("system::LocalTimeAppendLocalDateFunctionValue")

/**
* Appends a `Date` to a `Time` in order to return a `DateTime`.
*
*
* Note that the order in which the two objects are concatenated is irrelevant,
* so logically, `Date` + `Time`  produces the same result as `Time` + `Date`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `date` | A `Date`.
* | `time` | A `Time`, a time format that can include a time zone (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `Date` and `Time` objects to return a
* `DateTime`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "DateTime" : |2017-10-01| ++ |23:57:59-03:00|,
*   "DateTime2" : |2017-10-01| ++ |23:57:59Z|
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "DateTime": "2017-10-01T23:57:59-03:00",
*   "DateTime2": "2017-10-01T23:57:59Z"
* }
* ----
**/
fun ++(date: Date , time: Time): DateTime = native("system::LocalDateAppendTimeFunctionValue")


/**
* Appends a `Date` to a `Time` object to return a `DateTime`.
*
*
* Note that the order in which the two objects are concatenated is irrelevant,
* so logically, `Date` + `Time`  produces the same result as a `Time` + `Date`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `time` | A `Time`, a time format that can include a time zone (`Z` or `HH:mm`).
* | `date` | A `Date`.
* |===
*
* == Example
*
* This example concatenates `Time` and `Date` objects to return `DateTime`
* objects. Note that the first `LocalTime` `object is coerced to a `Time`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "DateTime1" : (|23:57:59| as Time) ++ |2017-10-01|,
*   "DateTime2" : |23:57:59Z| ++ |2017-10-01|,
*   "DateTime3" : |23:57:59+02:00| ++ |2017-10-01|
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "DateTime1": "2017-10-01T23:57:59Z",
*   "DateTime2": "2017-10-01T23:57:59Z",
*   "DateTime3": "2017-10-01T23:57:59+02:00"
* }
* ----
**/
fun ++(time: Time , date: Date): DateTime = native("system::TimeAppendLocalDateFunctionValue")

/**
* Appends a `TimeZone` to a `Date` type value and returns a `DateTime` result.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `date` | A `Date`.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `Date` and `TimeZone` (`-03:00`) to return a
* `DateTime`. Note the local time in the `DateTime` is `00:00:00` (midnight).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "DateTime" : (|2017-10-01| ++ |-03:00|) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "DateTime": "2017-10-01T00:00:00-03:00"
* }
* ----
*
**/
fun ++(date: Date , timezone: TimeZone): DateTime = native("system::LocalDateAppendTimeZoneFunctionValue")

/**
* Appends a `Date` to a `TimeZone` in order to return a `DateTime`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `date` | A `Date`.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `TimeZone` (`-03:00`) and `Date` to return a
* `DateTime`. Note the local time in the `DateTime` is `00:00:00` (midnight).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "DateTime" : |-03:00| ++ |2017-10-01| }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "DateTime": "2017-10-01T00:00:00-03:00"
* }
* ----
*
**/
fun ++(timezone: TimeZone , date: Date): DateTime = native("system::TimeZoneAppendLocalDateFunctionValue")

/**
* Appends a `TimeZone` to a `LocalDateTime` in order to return a `DateTime`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `dateTime` | A `LocalDateTime`, a date and time without a time zone.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `LocalDateTime` and `TimeZone` (`-03:00`) to return a
* `DateTime`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "DateTime" : (|2003-10-01T23:57:59| ++ |-03:00|) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "DateTime": "2003-10-01T23:57:59-03:00"
* }
* ----
**/
fun ++(dateTime: LocalDateTime , timezone: TimeZone): DateTime = native("system::LocalDateTimeAppendTimeZoneFunctionValue")

/**
* Appends a `LocalDateTime` to a `TimeZone` in order to return a `DateTime`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `dateTime` | A `LocalDateTime`, a date and time without a time zone.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `TimeZone` (`-03:00`) and `LocalDateTime` to return a
* `DateTime`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "TimeZone" : (|-03:00| ++ |2003-10-01T23:57:59|) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "TimeZone": "2003-10-01T23:57:59-03:00"
* }
* ----
**/
fun ++(timezone: TimeZone , datetime: LocalDateTime): DateTime = native("system::TimeZoneAppendLocalDateTimeFunctionValue")

/**
* Appends a `TimeZone` to a `LocalTime` in order to return a `Time`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `time` | A `LocalTime`, time format without a time zone.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `LocalTime` and `TimeZone` (`-03:00`) to return a
* `Time`. Note that the output returns`:00` for the unspecified seconds.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "Time" : (|23:57| ++ |-03:00|) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "Time": "23:57:00-03:00"
* }
* ----
*
**/
fun ++ (time: LocalTime, timezone: TimeZone): Time = native('system::LocalTimeAppendTimeZoneFunctionValue')

/**
* Appends a `LocalTime` to a `TimeZone` in order to return a `Time`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `time` | A `LocalTime`, a time format without a time zone.
* | `timezone` | A `TimeZone` (`Z` or `HH:mm`).
* |===
*
* == Example
*
* This example concatenates `TimeZone` (`-03:00`) and `LocalTime` to return a
* `Time`. Note that the output returns`:00` for the unspecified seconds.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "Time" : (|-03:00| ++ |23:57|) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "Time": "23:57:00-03:00"
* }
* ----
*
**/
fun ++ (timezone: TimeZone, time: LocalTime): Time = native('system::TimeZoneValueAppendLocalTimeFunctionValue')

/**
* Transforms each key-value pair of an object using the specified mapper function.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `object` | The object to transform.
* | `mapper` | Function that receives the `key`, `value` and `index` (optional) and returns the transformed key-value pair.
* |===
*
* == Example
*
* This example increases each price by 5 and formats the numbers to always include 2 decimals.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* {
*     prices: payload.prices mapObject (value, key) -> {
*         (key): (value + 5) as Number {format: "##.00"}
*     }
* }
* ----
*
* [[input_mapobject]]
* == Input
*
* [source,XML,linenums]
* ----
* <?xml version='1.0' encoding='UTF-8'?>
* <prices>
*     <basic>9.99</basic>
*     <premium>53</premium>
*     <vip>398.99</vip>
* </prices>
* ----
*
* [[output]]
* === Output
*
* [source,XML,linenums]
* ----
* <?xml version='1.0' encoding='UTF-8'?>
* <prices>
*   <basic>14.99</basic>
*   <premium>58.00</premium>
*   <vip>403.99</vip>
* </prices>
* ----
*
**/
fun mapObject <K,V>(object: {(K)?: V}, mapper : (value: V, key: K, index: Number) -> Object): Object = native('system::MapObjectObjectFunctionValue')

// Internal only. Do not publish this doc.
/**
* Helper function that allows `mapObject` to work with null values.
*/
fun mapObject(value: Null, mapper : (value: Nothing, key: Nothing, index: Number) -> Nothing): Null = null

/**
* Useful for mapping an object into an list (array), `pluck` iterates over an
* object and returns an array of keys, values, or indices in that object.
*
*
* It is an alternative to `mapObject`, which is similar but returns
* an object, instead of an array.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `object` | The object to map.
* | `mapper` | The `key`, `value`, and/or `index` (optional) used for mapping the input object.
* |===
*
* == Example
*
* This example uses `pluck` to iterate over each element (object) within
* `<prices/>` and returns arrays of their keys, values, and indices.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* var readXml = read("<prices>
*     <basic>9.99</basic>
*     <premium>53.00</premium>
*     <vip>398.99</vip>
*     </prices>", "application/xml")
* ---
* "result" : {
*   "keys" : readXml.prices pluck($$),
*   "values" : readXml.prices pluck($),
*   "indices" : readXml.prices pluck($$$)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "result": {
*     "keys": [
*       "basic",
*       "premium",
*       "vip"
*     ],
*     "values": [
*       "9.99",
*       "53",
*       "398.99"
*     ],
*     "indices": [
*       0,
*       1,
*       2
*     ]
*   }
* }
* ----
*
* == Example
*
* You can also use named keys and values as parameters. For example, the next
* transformation example iterates over the <<input_pluck, `prices` input>>
* above and outputs an array with a single element. Note that
* `payload pluck(payload.prices)` produces the same result as
* `payload pluck(payload[0])`.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* var readXml = read("<prices>
*     <basic>9.99</basic>
*     <premium>53.00</premium>
*     <vip>398.99</vip>
*     </prices>", "application/xml")
* ---
* payload pluck(readXml.prices)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   {
*     "basic": "9.99",
*     "premium": "53.00",
*     "vip": "398.99"
*   }
* ]
* ----
*/
fun pluck <K,V,R>(object: {(K)?: V}, mapper: (value: V,  key: K, index: Number) -> R): Array<R> = native('system::PluckObjectFunctionValue')

// Internal only. Do not publish this doc.
/**
* Helper function that allows *pluck* to work with null values.
*/
fun pluck(value: Null, mapper:(value: Nothing, key: Nothing, index: Nothing) -> Any): Null = null

/**
* Merges elements of two lists (arrays) into a single list (an array of arrays
* in consecutive `n`-tuples).
*
*
* Imagine two input lists each as one side of a zipper. Similar to the
* interlocking teeth of a zipper, the `zip` function interlocks each element
* from each input list, one element at a time.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `left` | The array input on the left-hand side of the function.
* | `right` | The array input on the right-hand side of the function.
* |===
*
* == Example
*
* This example interdigitates (zips together) elements of the left-hand and right-hand arrays.
* Notice that only elements with counterparts at the same index are returned
* in the array.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : [0, 1, 2, 3] zip ["a", "b", "c", "d"],
*   "b" : [0, 1, 2, 3] zip ["a"],
*   "c" : [0, 1, 2, 3] zip ["a", "b"],
*   "d" : [0, 1, 2] zip ["a", "b", "c", "d"]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": [
*     [0,"a"],
*     [1,"b"],
*     [2,"c"],
*     [3,"d"]
*     ],
*   "b": [
*     [0,"a"]
*   ],
*   "c": [
*     [0,"a"],
*     [1,"b"]
*   ],
*   "d": [
*     [0,"a"],
*     [1,"b"],
*     [2,"c"]
*   ]
* }
* ----
*
* == Example
*
* This example zips more than two input lists.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* var myvar = {
*    "list1": ["a", "b", "c", "d"],
*    "list2": [1, 2, 3],
*    "list3": ["aa", "bb", "cc", "dd"],
*    "list4": [["a", "b", "c"], [1, 2, 3, 4], ["aa", "bb", "cc", "dd"]]
* }
* ---
* myvar.list1 zip myvar.list2 zip myvar.list3 zip myvar.list4
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   [
*     [
*       ["a",1],
*       "aa"
*     ],
*     ["a","b","c"]
*   ],
*   [
*     [
*       ["b",2],
*       "bb"
*     ],
*     [1,2,3,4]
*   ],
*   [
*     [
*       ["c",3],
*       "cc"
*     ],
*     ["aa","bb","cc","dd"]
*   ]
* ]
* ----
**/
fun zip<T,R>(left: Array<T>, right: Array<R>): Array<Array<T | R>> =
   left match {
       case [lh ~ ltail] ->
         right match {
            case [rh ~ rtail] -> [[lh, rh] ~ zip(ltail, rtail)]
            case [] -> []
         }
       case [] -> []
   }

/**
* Transforms items from the given list (array) into a new list using the
* specified mapper function.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `items` | The list (`Array` type) to map.
* | `mapper` | Function used to transform each item in the list. It receives an `item` and optionally, an `index` and returns the transformed item.
* |===
*
* == Example
*
* This example concatenates the index (plus 1) to each value of the array.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* ['joe', 'pete', 'matt'] map ((item, index) -> (index + 1) ++ '_' ++ item)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   "1_joe",
*   "2_pete",
*   "3_matt"
* ]
* ----
*
* == Example
*
* This example creates an object with the array values, using the index (plus 1) as keys.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   (["joe", "pete", "matt"] map (item, index) -> {(index + 1): item})
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "1": "joe",
*   "2": "pete",
*   "3": "matt"
* }
* ----
*
*
* == Example
*
* If the parameters of the `mapper` function are not named, the index can be
* referenced with `&#36;&#36;`, and the value with `&#36;`.
* So the previous example can be written like this.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* ["joe", "pete", "matt"] map (($$ + 1) ++ ":" ++ upper($))
* ----
*
*
* == Example
*
* This example shows how to turn an array of key-value pairs into an object.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* ["joe", "pete", "matt"] map (($$ + 1) ++ ":" ++ upper($))
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   "1:JOE",
*   "2:PETE",
*   "3:MATT"
* ]
*
**/
fun map <T,R>(items: Array<T>, mapper: (item: T, index: Number) -> R ): Array<R> = native("system::ArrayMapFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows `map` to work with null values.
*/
fun map(value: Null, mapper: (item: Nothing, index: Nothing) -> Any): Null = null

/**
* Applies a transformation function to each element in a list (array) and
* then flattens the result.
*
*
* Instead of returning an array of arrays (as `map` does), it returns a
* flattened array (see the `flatten` function).
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `items` | The list (`Array` type) to map.
* | `mapper` | The `item` and/or `index` in the list to flatten.
* |===
*
* == Example
*
* This example returns an array containing each index and item in order. Note
* that you can also write the example like this:
* `{ "users" : ["joe", "pete", "matt"] flatMap ([$$ as String, $]) }`
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "users" : ["joe", "pete", "matt"] flatMap (item, index) -> ([index as String, item]) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "users": [
*      "0",
*      "joe",
*      "1",
*      "pete",
*      "2",
*      "matt"
*    ]
*  }
* ----
*/
fun flatMap<T,R>(items: Array<T>, mapper:(item: T, index: Number) -> Array<R>): Array<R> =
    flatten(items map (value,index) -> mapper(value,index))

// Internal only. Do not publish this doc.
/**
* Helper function that allows `flatMap` to work with null values.
*/
fun flatMap<T,R>(value: Null, mapper: (item: Nothing, index: Nothing) -> Any): Null = null

/**
* Filters a list (array) by applying an expression that returns only the matching
* items from the list.
*
*
* The expression must return `true` or `false`. If the expression returns `true`
* for an element, the element remains in the list. If it returns `false` for
* an element, the element gets filtered out of the results.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `items` | The list (array) to filter.
* | `criteria` | Function that receives an `item` and `index` and returns true if the element should be kept.
* |===
*
* == Example
*
* This example returns an array of all elements greater than `2`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* [9,2,3,4,5] filter (myitem, myindex) -> (myitem > 2)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [9,3,4,5]
* ----
*
* == Example
*
* This example returns an array of all elements found at an index
* greater than `2`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* [9,2,3,4,5] filter ((myitem, myindex) -> (myindex > 2))
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [4,5]
* ----
*
* == Example
*
* This example returns an array of all elements found at an index (`&#36;&#36;`)
* greater than `1` where the value of the element is less than `5`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* [9, 2, 3, 4, 5] filter (($$ > 1) and ($ < 5))
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [3,4]
* ----
**/
fun filter <T,T>(items: Array<T> , criteria: (item: T, index: Number) -> Boolean): Array<T> = native("system::ArrayFilterFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows `filter` to work with null values.
*/
fun filter(value: Null, criteria: (item: Nothing, index: Nothing) -> Boolean): Null = null

/**
* Filters an object, keeping the key-value pairs that fulfill the `criteria`.
*
*
* The criteria (a lambda expression) has three parameters: `value`, `key`, and
* `index`. You can reference the value with `&#36;`, the key with `&#36;&#36;`,
* and the index with `&#36;&#36;&#36;`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The input object.
* | `criteria` | The expression that determines whether to retain the key-value pair or not.
* |===
*
* == Example
*
* This example keeps only the key-value pairs if the value equals `"a"`.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {"letter1": "a", "letter2": "b"} filterObject ((value) -> value == "a")
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "letter1": "a"
* }
* ----
*
* == Example
*
* This example only keeps the key-value pairs where the key starts with "letter".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {"letter1": "a", "letter2": "b", "id": 1} filterObject ((value, key) -> key startsWith "letter")
* ----
*/
fun filterObject <K,V>(value: {(K)?: V}, criteria: (value: V, key: K, index: Number) -> Boolean): {(K)?: V} = native("system::ObjectFilterFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows `filterObject` to work with null values.
*/
fun filterObject(value: Null, criteria: (value: Nothing, key: Nothing, index: Nothing) -> Boolean): Null = null

/**
* Replaces the part of a string that matches a regular expression and requires
* the use of `with` to specify the replacement.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string (`String` type).
* | `matcher` | A regular expression for matching characters in the `text`.
* |===
*
* == Example
*
* This example replaces the numbers at the end of a string with different
* characters. Note that you can also use this notation
* `replace(text,matcher) with string` (for example,
* `replace("admin123", /(\d+)/) with("ID")`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "replace" : "admin123" replace /(\d+)/ with "ID" }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "replace": "adminID"
* }
* ----
**/
fun replace(text: String, matcher: Regex): ((Array<String>, Number) -> String) -> String = native("system::ReplaceStringRegexFunctionValue")

/**
* Replaces part of a string with another string.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string (`String` type).
* | `matcher` | A string (`String` type) for matching characters in the `text`.
* |===
*
* == Example
*
* This example replaces the numbers at the end of a string with different
* characters.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "replace": "admin123" replace "123" with "ID" }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "replace": "adminID"
* }
* ----
**/
fun replace(text: String, matcher: String): ((Array<String>, Number) -> String) -> String = native("system::ReplaceStringStringFunctionValue")

/**
* When used with `replace`, `with` passes a replacement.
*
*
* == Example
*
* This example replaces all numbers in a string with "x" characters.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "ssn" : "987-65-4321" replace /[0-9]/ with("x") }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "ssn": "xxx-xx-xxxx" }
* ----
*/
fun with<V,U,R,X>(toBeReplaced: ((V, U) -> R) -> X, replacer: (V, U) -> R ): X = toBeReplaced(replacer)


/**
* Applies the reduction function for each element in the input array.
*
*
* Note that if the array is empty and no default value is set on the
* accumulator, a null value is returned.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `item` | Item in the given list. It provides the value to reduce. Can also be referenced as `&#36;`.
* | `acc` | An accumulator (also referenced as `&#36;&#36;`) to apply to items in the input array. It starts with the first value of the array by default.
* |===
*
* == Example
*
* This example returns the sum of the values in the input arrays.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*  {
*     "sum" : [0, 1, 2, 3, 4, 5] reduce ($$ + $),
*     "sum" : [0, 1, 2, 3, 4, 5] reduce ((val, acc) -> acc + val)
*  }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "sum": 15,
*   "sum": 15
* }
* ----
*
* == Example
*
* This example uses the accumulator to concatenate elements in the input
* arrays and return the results in a string.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "concat" : ["a", "b", "c", "d"] reduce ($$ ++ $),
*    "concat" : ["a", "b", "c", "d"] reduce ((val, acc) -> acc ++ val)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "concat": "abcd",
*   "concat": "abcd"
* }
* ----
*
* == Example
*
* This example sets the first elements of the arrays to `"z"` and `3`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "concat" : ["a", "b", "c", "d"] reduce ((item, acc = "z") -> acc ++ item),
*    "sum": [0, 1, 2, 3, 4, 5] reduce ((val, acc = 3) -> acc + val)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "concat": "zabcd"
*   "sum": 18
* }
* ----
*
* == Example
*
* This example shows a variety of uses of `reduce`, including its application to
* arrays of boolean values and objects.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* var in0 =
* {
*   "a": [0, 1, 2, 3, 4, 5],
*   "b": ["a", "b", "c", "d", "e"],
*   "c": [{ "letter": "a" }, { "letter": "b" }, { "letter": "c" }],
*   "d": [true, false, false, true, true]
* }
* ---
* {
*   "a" : [0, 1, 2, 3, 4, 5] reduce $$,
*   "b": ["a", "b", "c", "d", "e"] reduce $$,
*   "c": [{ "letter": "a" }, { "letter": "b" }, { "letter": "c" }] reduce ((val, acc = "z") -> acc ++ val.letter),
*   "d": [{ letter: "a" }, { letter: "b" }, { letter: "c" }] reduce $$,
*   "e": [true, false, false, true, true] reduce ($$ and $),
*   "f": [true, false, false, true, true] reduce ((val, acc) -> acc and val),
*   "g": [true, false, false, true, true] reduce ((val, acc = false) -> acc and val),
*   "h": [true, false, false, true, true] reduce $$,
*   "i": in0.a reduce ($$ + $),
*   "j": in0.a reduce ((val, acc) -> acc + val),
*   "k": in0.a reduce ((val, acc = 3) -> acc + val),
*   "l": in0.a reduce $$,
*   "m": in0.b reduce ($$ ++ $),
*   "n": in0.b reduce ((val, acc) -> acc ++ val),
*   "o": in0.b reduce ((val, acc = "z") -> acc ++ val),
*   "p": in0.b reduce $$,
*   "q": in0.c reduce ((val, acc = "z") -> acc ++ val.letter),
*   "r": in0.c reduce $$,
*   "s": in0.d reduce ($$ and $),
*   "t": in0.d reduce ((val, acc) -> acc and val),
*   "u": in0.d reduce ((val, acc = false) -> acc and val),
*   "v": in0.d reduce $$,
*   "w": ([0, 1, 2, 3, 4] reduce ((val, acc = {}) -> acc ++ { a: val })) pluck $,
*   "x": [] reduce $$,
*   "y": [] reduce ((val,acc = 0) -> acc + val)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* "a": 0,
* "b": "a",
* "c": "zabc",
* "d": { "letter": "a" },
* "e": false,
* "f": false,
* "g": false,
* "h": true,
* "i": 15,
* "j": 15,
* "k": 18,
* "l": 0,
* "m": "abcde",
* "n": "abcde",
* "o": "zabcde",
* "p": "a",
* "q": "zabc",
* "r": { "letter": "a" },
* "s": false,
* "t": false,
* "u": false,
* "v": true,
* "w": [ 0,1,2,3,4 ],
* "x": null,
* "y": 0
* }
* ----
**/
fun reduce <T>(items: Array<T>, callback: (item: T, accumulator: T) -> T ): T | Null = native("system::ArrayReduceFunctionValue")
//Works like fold left
fun reduce <T,A>(items: Array<T>, callback: (item: T, accumulator: A) -> A ): A = native("system::ArrayReduceFunctionValue")

/**
* Classifies the elements of a list (array) using the specified `criteria` function.
*
* The resulting object will have the grouping criteria as keys.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `items` | The list (`Array` type) to classify in groups.
* | `criteria` | Function that receives the `item` and `index` of the element to group and should return the criteria to group by.
* |===
*
* == Example
*
* This example groups the elements of an array based on the language field.
* Notice that it uses the `item.language` selector to specify the criteria.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* payload groupBy (item) -> item.language
* ----
*
* === Input
*
* [source,JSON,linenums]
* ----
* [
*     {
*       "name": "Foo",
*       "language": "Java"
*     },
*     {
*       "name": "Bar",
*       "language": "Scala"
*     },
*     {
*       "name": "FooBar",
*       "language": "Java"
*     }
* ]
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "Scala": [
*     {
*       "name": "Bar",
*       "language": "Scala"
*     }
*   ],
*   "Java": [
*     {
*       "name": "Foo",
*       "language": "Java"
*     },
*     {
*       "name": "FooBar",
*       "language": "Java"
*     }
*   ]
* }
* ----
*/
fun groupBy <T,R>(items: Array<T> , criteria: (item: T, index: Number) -> R): {(R): Array<T>} = native("system::ArrayGroupByFunctionValue")

/**
* Groups elements of an object based on a supplied key.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `object` | The object to group.
* | `criteria` | The `key` and/or `value` of the object to use for grouping.
* |===
*
*
* == Example
*
* The example groups an object based on its value (`$$`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "a" : "b"} groupBy $
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "b": {
*     "a": "b"
*   }
* }
* ----
*/
fun groupBy <K,V,R>(object: {(K)?: V}, criteria: (value: V, key: K) -> R): {(R): {(K)?: V}} = native("system::ObjectGroupByFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *groupBy* to work with null values.
*/
fun groupBy(value: Null, criteria: (Nothing, Nothing) -> Any): Null = null

//REMOVE
/**
* Removes specified items from a list (an array).
*
*
* |===
* | Name   | Description

* | `source` | The list (an `Array` type).
* | `toRemove` | Items to remove from the list.
* |===
*
* == Example
*
* This example removes a items from a list.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "a" : [0, 1, 1, 2] -- [1,2] }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": [0]
* }
* ----
**/
fun -- <S>(source: Array<S> , toRemove: Array<Any>): Array<S> = native("system::ArrayRemoveFunctionValue")

/**
* Removes specified key-value pairs from an object.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `source` | The object.
* | `toRemove` | Objects to remove from the source object.
* |===
*
* == Example
*
* This example removes a key-value pair from the source object.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "hello" : "world",
*    "name" : "DW"
* } -- { "hello" : "world"}
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "name": "DW"
* }
* ----
*/
fun -- <K,V>(source: {(K)?: V} , toRemove: Object): {(K)?:V} = native("system::ObjectRemoveFunctionValue")

/**
 * Removes specified key-value pairs from an object.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name   | Description
 * | `source` | The source object (an `Object` type).
 * | `toRemove` | Keys for the key-value pairs to remove from the source object.
 * |===
 *
 * == Example
 *
 * This example removes two key-value pairs from the source object.
 *
 * === Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * output application/json
 * ---
 * {
 *    "yes" : "no",
 *    "good" : "bad",
 *    "old" : "new"
 * } -- ["yes", "old"]
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    "good": "bad"
 * }
 * ----
 */
fun --(source: Object, keys: Array<String>) =
  keys reduce (key, obj = source) -> (obj - key)

/**
 * Removes specified key-value pairs from an object.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name   | Description
 * | `source` | The source object (an `Object` type).
 * | `keys` | A keys for the key-value pairs to remove from the source object.
 * |===
 *
 * == Example
 *
 * This example specifies the key-value pair to remove from the source object.
 *
 * === Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * output application/json
 * ---
 * {
 *    "hello" : "world",
 *    "name" : "DW"
 *  } -- ["hello" as Key]
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    "name": "DW"
 * }
 * ----
 */
fun --(source: Object, keys: Array<Key>) =
  keys reduce (key, obj = source) -> (obj - key)

/**
* Returns indices of the input array (a list) that match a specified `elementToFind`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `elements` | An array with elements of any type.
* | `elementToFind` | Value to find in the input array.
* |===
*
* == Example
*
* This example finds the index of an element in a string array.
*
* === Source
*
* [source,DataWeave,lineums]
* ----
* %dw 2.0
* output application/json
* ---
* ["Bond", "James", "Bond"] find "Bond"
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [0,2]
* ----
*
*/
fun find <T>(elements: Array<T> , elementToFind: Any): Array<Number> = native("system::ArrayFindFunctionValue")

/**
* Returns the indices in the text that match the specified regular expression
* (regex) followed by the capture groups.
*
*
* The first element in each resulting sub-array is the index in the text that
* matches the regex, and the next ones are the capture groups in the regex
* (if present).
*
* Note: To retrieve parts of the text that match a regex use the scan function.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string (`String` type).
* | `matcher` | A regular expression for matching characters in the `text`.
* |===
*
* == Example
*
* This example finds the beginning and ending indices of words that contain `ea`
*
* === Source
*
* [source,DataWeave,lineums]
* ----
* %dw 2.0
* output application/json
* ---
* "I heart DataWeave" find /\w*ea\w*(\b)/
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*    [2,7], [8,17]
* ]
* ----
*/
fun find(text: String , matcher: Regex): Array<Array<Number>> = native("system::StringFindRegexFunctionValue")

/**
* Lists indices where the specified characters of a string are present.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A source string.
* | `textToFind` | The string to find in the source string.
* |===
*
* == Example
*
* This example lists the indices of "a" found in "aabccdbce".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* "aabccdbce" find "a"
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [0,1]
* ----
*
**/
fun find(text: String, textToFind: String): Array<Number> = native("system::StringFindStringFunctionValue")

/**
* Returns unique values from a list (array) that might have duplicates.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `items` | The list (`Array` type).
* | `criteria` | The `item` and/or `index` used to return.
* |===
*
* == Example
*
* This example removes duplicates of `"Kurt Cagle"` from an input array.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* var record =  {
*   "title": "XQuery Kick Start",
*   "author": [
*     "James McGovern",
*     "Per Bothner",
*     "Kurt Cagle",
*     "James Linn",
*     "Kurt Cagle",
*     "Kurt Cagle",
*     "Kurt Cagle",
*     "Vaidyanathan Nagarajan"
*   ],
*   "year":"2000"
* }
* ---
* {
*     "book" : {
*       "title" : record.title,
*       "year" : record.year,
*       "authors" : record.author distinctBy $
*     }
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "book": {
*     "title": "XQuery Kick Start",
*     "year": "2000",
*     "authors": [
*       "James McGovern",
*       "Per Bothner",
*       "Kurt Cagle",
*       "James Linn",
*       "Vaidyanathan Nagarajan"
*     ]
*   }
* }
* ----
**/
fun distinctBy <T>(items: Array<T>, criteria: (item: T, index: Number) -> Any): Array<T> = native("system::ArrayDistinctFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *distinctBy* to work with null values.
*/
fun distinctBy(items: Null, criteria: (item: Nothing, index: Nothing) -> Any): Null = null

/**
* Removes duplicate key-value pairs from an Object.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `object` | The object from which to remove the key-value pairs.
* | `criteria` | The `key` and/or `value` used to identify the key-value pairs to remove.
* |===
*
* == Example
*
* This example removes duplicates (`<author>James McGovern</author>`)
* from `<book/>`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/xml
* ---
* {
*    book : {
*      title : payload.book.title,
*      authors: payload.book.&author distinctBy $
*    }
* }
* ----
*
* === Input
*
* [source,XML,linenums]
* ----
* <book>
*   <title> "XQuery Kick Start"</title>
*   <author>James Linn</author>
*   <author>Per Bothner</author>
*   <author>James McGovern</author>
*   <author>James McGovern</author>
*   <author>James McGovern</author>
* </book>
* ----
*
* === Output
*
* [source,XML,linenums]
* ----
* <book>
*   <title> "XQuery Kick Start"</title>
*   <authors>
*       <author>James Linn</author>
*       <author>Per Bothner</author>
*       <author>James McGovern</author>
*   </authors>
* </book>
* ----
*
**/
fun distinctBy <K, V>(object: {(K)?: V}, criteria: (value: V, key: K) -> Any): Object = native("system::ObjectDistinctFunctionValue")

/**
* Returns a range with the specified boundaries.
*
*
* The upper boundary is inclusive.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `from` | A number (`Number` type) that starts the range.
* | `to` | A number (`Number` type) that ends the range.
* |===
*
* == Example
*
* This example lists a range of numbers from 1 to 10.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*     "myRange": 1 to 10
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*     "myRange": [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]
* }
* ----
*/
fun to(from: Number , to: Number): Range = native("system::ToRangeFunctionValue")

//CONTAINS
/**
* Returns `true` if a list (array) contains a given value, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `items` | The input list (an `Array`).
* | `elements` | An element to find in the list. Can be any supported data type.
* |===
*
* == Example
*
* This example indicates whether the input list contains '"3"'.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* ContainsRequestedItem: payload.root.*order.*items contains "3"
* ----
*
* === Input
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <root>
*     <order>
*       <items>155</items>
*     </order>
*     <order>
*       <items>30</items>
*     </order>
*     <order>
*       <items>15</items>
*     </order>
*     <order>
*       <items>5</items>
*     </order>
*     <order>
*       <items>4</items>
*       <items>7</items>
*     </order>
*     <order>
*       <items>1</items>
*       <items>3</items>
*     </order>
*     <order>
*         null
*     </order>
* </root>
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "ContainsRequestedItem": true
* }
* ----
**/
fun contains <T>(items: Array<T> , element: Any): Boolean = native("system::ArrayContainsFunctionValue")

/**
* Indicates whether a string contains a given substring. Returns `true`
* or `false`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | An input string (a `String`).
* | `toSearch` | The substring (a `String`) to find in the input string.
* |===
*
* == Example
*
* This example finds the substring `"me"` in `"some string"`, so it returns `true`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "ContainsString" : payload.root.mystring contains "me" }
* ----
*
* === Input
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <root>
*   <mystring>some string</mystring>
* </root>
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "ContainsString": true }
* ----
**/
fun contains(text: String , toSearch: String): Boolean = native("system::StringStringContainsFunctionValue")

/**
* Returns `true` if a string contains a match to a regular expression, `false`
* if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | An input string (`String` type).
* | `matcher` | A regular expression for matching characters in the input `text`.
* |===
*
* == Example
*
* This example finds a match to `/s[t|p]rin/` within `"A very long string"`,
* so it returns `true`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* ContainsString: payload.root.mystring contains /s[t|p]rin/
* ----
*
* === Input
*
* [source,XML,linenums]
* ----
* <?xml version="1.0" encoding="UTF-8"?>
* <root>
*   <mystring>A very long string</mystring>
* </root>
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "ContainsString": true
* }
* ----
**/
fun contains(text: String , matcher: Regex): Boolean = native("system::StringRegexContainsFunctionValue")


//ORDERBY
// sduke tweak: more to do later
/**
* Reorders the content of an object using a value returned by a function as the criteria.
*
*
* Note that you can reference the index with
* `&#36;&#36;` and the value with `&#36;`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `object` | The object to reorder.
* | `criteria` | The result of the function is used as the criteria to reorder the object.
* |===
*
* == Example
*
* This example alphabetically orders the values of each object in the array.
* Note that `orderBy($.letter)` produces the same result as `orderBy($[0])`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* orderByLetter: [{ letter: "d" }, { letter: "e" }, { letter: "c" }, { letter: "a" }, { letter: "b" }] orderBy($.letter)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "orderByLetter": [
*     {
*       "letter": "a"
*     },
*     {
*       "letter": "b"
*     },
*     {
*       "letter": "c"
*     },
*     {
*       "letter": "d"
*     },
*     {
*       "letter": "e"
*     }
*   ]
* }
* ----
*
* == Example
*
* The `orderBy` function does not have an option to order in descending order
* instead of ascending. In these cases, you can simply invert the order of
* the resulting array using `-`, for example:
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* orderDescending: ([3,8,1] orderBy -$)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "orderDescending": [8,3,1] }
* ----
**/
fun orderBy <K,V,R, O <: {(K)?: V}>(object: O, criteria: (value: V, key: K) -> R): O = native('system::ObjectOrderByFunctionValue')

// sduke: tweak
/**
* Sorts an array using the specified criteria.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `array` | The list (an array) to sort.
* | `criteria` | The result of the function will be used as the criteria to sort the list. It should return a simple value (String, Number, etc)
* |===
*
* == Example
*
* This example sorts an array of numbers based on the numeric values.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* [3,2,3] orderBy $
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   2,
*   3,
*   3
* ]
* ----
*
* == Example
*
* This example sorts an array of people based on their age.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* [{name: "Santiago", age: 42},{name: "Leandro", age: 29}, {name: "Mariano", age: 35}] orderBy (person) -> person.age
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   {
*     name: "Leandro",
*     age: 29
*   },
*   {
*     name: "Mariano",
*     age: 35
*   },
*   {
*     name: "Santiago",
*     age: 42
*   }
* ]
* ----
**/
fun orderBy <T,R>(array: Array<T> , criteria: (item: T, index: Number) -> R): Array<T> = native("system::ArrayOrderByFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *orderBy* to work with null values.
*/
fun orderBy(value: Null , criteria: (item: Nothing, index: Nothing) -> Null): Null = null

//UNARY OPERATORS
/**
* Returns the average of numeric values in a list (an array).
*
*
* A list that is empty or that contains a non-numeric value results
* in an error.
*
* == Example
*
* This example returns the average of multiple arrays.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: avg([1, 1000]),
*   b: avg([1, 2, 3])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 500.5,
*   "b": 2.0
* }
* ----
*
**/
fun avg(values: Array<Number>): Number = sum(values) / sizeOf(values)

/**
* Returns the highest numeric value in a list (an array).
*
*
* Returns null if the array is empty and produces an error when non-numeric
* values are in the array.
*
* == Example
*
* This example returns the maximum value of each input array.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: max([1, 1000]),
*   b: max([1, 2, 3]),
*   d: max([1.5, 2.5, 3.5])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 1000,
*   "b": 3,
*   "d": 3.5
* }
* ----
**/
fun max <T <: Comparable>(values: Array<T>): T | Null = values maxBy $

/**
* Returns the lowest numeric value in an array.
*
*
* Returns null if the array is empty and produces an error when non-numeric
* values are in the array.
*
* == Example
*
* This example returns the lowest numeric value of each input array.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: min([1, 1000]),
*   b: min([1, 2, 3]),
*   d: min([1.5, 2.5, 3.5])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 1,
*   "b": 1,
*   "d": 1.5
* }
* ----
**/
fun min <T <: Comparable>(values: Array<T>): T | Null = values minBy $

/**
* Returns the sum of numbers in an array.
*
*
* Returns `0` if the array is empty and produces an error when non-numeric
* values are in the array.
*
* == Example
*
* This example returns the sum of the values in the input array.
*
* === Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* sum([1, 2, 3])
* ----
*
* === Output
* [source,JSON,linenums]
* ----
* 6
* ----
*/
fun sum(values: Array<Number>): Number = (values reduce (value, acc) -> value + acc) default 0

//SIZEOF
/**
* Returns the number of elements in an array.
*
*
* Returns `0` if the array is empty.
*
* == Example
*
* This example returns a count of elements in the input array.
*
* === Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "arraySize": sizeOf([1,2,3])
* }
* ----
*
* === Output
* [source,JSON,linenums]
* ----
* {
*   "arraySize": 3
* }
* ----
**/
fun sizeOf(array: Array<Any>): Number = native("system::ArraySizeOfFunctionValue")

/**
* Returns the number of key-value pairs in an input object.
*
*
* Returns `0` if the object is empty.
*
* == Example
*
* This example counts the key-value pairs in an input object.
*
* === Source
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   objectSize: sizeOf({a:1,b:2})
* }
* ----
*
* === Output
* [source,JSON,linenums]
* ----
* {
*   "objectSize": 2
* }
* ----
**/
fun sizeOf(array: Object): Number = native("system::ObjectSizeOfFunctionValue")

/**
* Returns the byte length of a binary.
*
*
* == Example
*
* This example returns the size of a binary value that is passed through a
* variable. The binary is returned by a function in `core::Binaries`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Binaries
* var var1 = fromBase64(000000)
* output application/json
* ---
*
* { "size" : sizeOf(var1) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "size": 4
* }
* ----
**/
fun sizeOf(array: Binary): Number = native("system::BinarySizeOfFunctionValue")

/**
* Returns the number of characters (including white space) in an string.
*
*
* Returns `0` if the string is empty.
*
* == Example
*
* This example returns the number of characters in the input strings.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "sizeOfString1" : sizeOf("MuleSoft"),
*   "sizeOfSting2" : sizeOf("my string")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "sizeOfString1": 8,
*   "sizeOfSting2": 9
* }
* ----
**/
fun sizeOf(array: String): Number = native("system::StringSizeOfFunctionValue")

/**
* Flattens an array of arrays into a single, simple array.
*
*
* == Example
*
* This example flattens an array of arrays.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* flatten(payload)
* ----
*
* === Input
*
* [source,JSON,linenums]
* ----
* [
*    [3,5],
*    [9,5],
*    [154,0.3]
* ]
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* [
*   3,
*   5,
*   9,
*   5,
*   154,
*   0.3
* ]
* ----
*
**/
fun flatten <T, Q>(items: Array<Array<T> | Q>): Array<T | Q> = native("system::ArrayFlattenFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *flatten* to work with null values.
*/
fun flatten (value: Null): Null = null

/**
* Performs the opposite of `zip`.
*
*
* Given a single array where each index contains an array with two elements,
* `unzip` outputs two separate arrays, each with the corresponding elements
* of each pair. If the indices in the provided array contain arrays with more
* than two elements, the output contains as many arrays as there are
* elements for each index.
*
* == Example
*
* This example unzips an array of arrays. Note that in example `b`, the number
* of elements in input array is not consistent. So the function only creates
* as many full arrays as it can, in this case just one.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : unzip([[0,"a"],[1,"b"],[2,"c"],[3,"d"]]),
*   "b" : unzip([ [0,"a"], [1,"a","foo"], [2], [3,"a"]])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "a":[
*       [0, 1, 2, 3],
*       ["a", "b", "c", "d"]
*     ],
*   "b": [
*       [0,1,2,3]
*     ]
* }
* ----
*
**/
fun unzip<T>(items: Array<Array<T>>):Array<Array<T>> = do {
    var minSize = min(items map sizeOf($)) default 0
    ---
    ((0 to minSize - 1) as Array<Number>) map ((i)-> items map (item) -> item[i])
}

/**
* Returns `true` if a string ends with a provided substring, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The input string (a `String`).
* | `suffix` | The suffix string to find at the end of the input string.
* |===
*
* == Example
*
* This example finds "no" (but not "to") at the end of "Mariano".
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "yes" : "Mariano" endsWith "no",
*   "no" : "Mariano" endsWith "to"
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "yes": true,
*   "no": false
* }
* ----
**/
fun endsWith(text: String, suffix: String): Boolean = native("system::StringEndsWithFunctionValue")

/**
* Merges a list (array) into a single string value and uses the provided string
* as a separator between each item in the list.
*
*
* Note that `joinBy` performs the opposite task of `splitBy`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `elements` |  The list (an `Array`).
* | `separator` | A `String` used to join elements in the list.
* |===
*
* == Example
*
* This example joins the elements with a hyphen (`-`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* "hyphenate" : ["a","b","c"] joinBy "-"
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "hyphenate": "a-b-c"
* }
* ----
**/
fun joinBy(elements: Array<Any>, separator: String): String = native("system::ArrayJoinFunctionValue")

/**
* Returns a list (array) with all of the matches found within the given string.
*
*
* Each match is returned as an array that contains the complete match followed by any
* capture groups in your regular expression (if present).
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The text to scan (`String` type).
* | `regex` | A regular expression that describes the pattern to look for in the text.
* |===
*
* == Example
*
* In the example, the `regex` describes an email address. It contains two
* capture groups, the characters before and after the `@`. It produces an
* an array matching the two email addresses in the input string. Each match
* is an array of three elements: The first is the entire email address, and the
* second and third are matches to the `regex` capture groups.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* "hello" : "anypt@mulesoft.com,max@mulesoft.com" scan /([a-z]*)@([a-z]*).com/
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "hello": [
*     [
*       "anypt@mulesoft.com",
*       "anypt",
*       "mulesoft"
*     ],
*     [
*       "max@mulesoft.com",
*       "max",
*       "mulesoft"
*     ]
*   ]
* }
* ----
**/
fun scan(text: String, matcher: Regex): Array<Array<String>> = native("system::StringScanFunctionValue")

/**
* Splits a string based on a regex.
*
*
* The separator can match any character in the input. Note that `splitBy`
* performs the opposite operation of `joinBy`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split.
* | `regex` | A regular expression used to separate string. If it does not match some part of the string, the function will return the string unseparated in an array.
* |===
*
* == Example
*
* This example uses the regular expression `\/^*.b./\` to find and use `-b-` as
* a separator. Notice that the separator is omitted from the output.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "split" : "a-b-c" splitBy(/^*.b./) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "split": ["a","c"]
* }
* ----
**/
fun splitBy(text: String, regex: Regex): Array<String> = native("system::StringSplitStringFunctionValue")

/**
* Splits a string based on a separator.
*
*
* The separator can match any character in the input. Note that `splitBy` performs
* the opposite operation of `joinBy`.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split.
* | `separator` | A string used to separate the input string. It must match some part of the string.
* |===
*
* == Example
*
* This example uses the hyphen (`-`) as the separator. Note that the selector is
* not retained in the output.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "split" : "a-b-c" splitBy("-") }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "split": ["a", "b", "c"]
* }
* ----
**/
fun splitBy(text: String, separator: String): Array<String> = native("system::StringSplitRegexFunctionValue")

/**
* Returns `true` or `false` depending on whether the input string starts with a
* matching prefix.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The input string.
* | `prefix` | A string that identifies the prefix.
* |===
*
* == Example
*
* This example indicates whether the strings start with a given prefix. Note that
* you can also use the `startsWith(text,prefix)` notation (for example,
* `startsWith("Mariano","Mar")`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "yes" : "Mariano" startsWith "Mar",
*   "no" : "Mariano" startsWith "Em"
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "yes": true,
*   "no": false
* }
* ----
**/
fun startsWith(text: String, prefix: String): Boolean = native("system::StringStartsWithFunctionValue")

/**
* Checks if the given matcher matches the (whole) text.
*
*
* For use cases where you need to output or conditionally process the matched
* value, see
* https://docs.mulesoft.com/mule-runtime/4.1/dataweave-pattern-matching[Pattern Matching in DataWeave].
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The input string.
* | `matcher` | A regular expression for matching characters in the string.
* |===
*
* == Example
*
* This example indicates whether the regular expression matches the input text.
* Note that you can also use the `matches(text,matcher)` notation (for example,
* `matches("admin123", /a.*\d+/)`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "match" : "admin123" matches /a.*\d+/,
*    "nonmatch" : "admin123" matches /^b.+/
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*     "match": true,
*     "nonmatch": false
* }
* ----
**/
fun matches(text: String, matcher: Regex): Boolean = native("system::StringMatchesFunctionValue")

/**
* Uses an regular expression to match string and then separate it into capture
* groups. Returns the results in a a list (an array).
*
*
* It can be applied to the result of any evaluated expression and can return
* any evaluated expression. See
* https://docs.mulesoft.com/mule-runtime/4.1/dataweave-pattern-matching[Pattern Matching in DataWeave].
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string (`String` type).
* | `matcher` | A regular expression for matching characters in the `text`.
* |===
*
* == Example
*
* In this example, the regular expression describes an email address. It
* contains two capture groups: what come before and after the `@`. The
* result is an array of three elements: the first is the whole email address,
* the second matches one of the capture groups, the third matches the other one.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* { "hello" : "anniepoint@mulesoft.com" match(/([a-z]*)@([a-z]*).com/) }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "hello": [
*     "anniepoint@mulesoft.com",
*     "anniepoint",
*     "mulesoft"
*   ]
* }
* ----
**/
fun match(text: String, matcher: Regex): Array<String> = native("system::StringRegexMatchFunctionValue")

/**
* Returns the provided string in lowercase characters.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string (`String` type).
* |===
*
* == Example
*
* This example converts uppercase characters to lower-case.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "name" : lower("MULESOFT")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "name": "mulesoft"
* }
* ----
**/
fun lower(text: String): String = native("system::StringLowerFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *lower* to work with null values.
*/
fun lower(value:Null) : Null = null

/**
* Removes any blank spaces from the beginning and ending of a string.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string from which to remove any blank spaces.
* |===
*
* == Example
*
* This example trims a string.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "trim": trim("   my long text     ")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "trim": "my long text"
* }
* ----
**/
fun trim(text: String): String = native("system::StringTrimFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *trim* to work with null values.
*/
fun trim(value: Null): Null = null

/**
* Returns the provided string in upper-case characters.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to convert to uppercase.
* |===
*
* == Example
*
* This example converts lowercase characters to uppercase.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "name" : upper("mulesoft")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "name": "MULESOFT"
* }
* ----
**/
fun upper(text: String): String = native("system::StringUpperFunctionValue")

// Internal only. Do not publish this doc.
/**
* Helper function that allows *trim* to work with null values.
*/
fun upper(value: Null): Null = null

/**
* Raises the value of a given `base` number to the specified `power`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `base` |  A number (`Number` type) that serves as the base.
* | `power` |  A number (`Number` type) that serves as the power.
* |===
*
* == Example
*
* This example raises the value of the `base` number to the specified `power`.
* Note that you can also use the `pow(base,power)` notation (for example,
* `pow(2,3)` to return `8`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: 2 pow 3,
*   b: 3 pow 2,
*   c: 7 pow 3
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 8,
*   "b": 9,
*   "c": 343
* }
* ----
**/
fun pow(base: Number, power: Number): Number = native("system::PowNumberFunctionValue")

/**
* Returns the modulo (the remainder after performing dividing the `dividend`
* by the `divisor`).
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `dividend` |  The number that serves as the dividend for the operation.
* | `divisor` |  The number that serves as the divisor for the operation.
* |===
*
* == Example
*
* This example returns the modulo of the input values. Note that you can also
* use the `mod(dividend, divisor)` notation (for example, `mod(3, 2)` to return
* `1`).
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : 3 mod 2,
*   "b" : 4 mod 2,
*   "c" : 2.2 mod 2
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 1,
*   "b": 0,
*   "c": 0.2
* }
* ----
**/
fun mod(dividend: Number, divisor: Number): Number = native("system::ModuleNumberFunctionValue")

/**
* Returns the square root of an input number.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to apply the operation to.
* |===
*
* == Example
*
* This example returns the square root of an input numbers.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : sqrt(4),
*   "b" : sqrt(25),
*   "c" : sqrt(100)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 2.0,
*   "b": 5.0,
*   "c": 10.0
* }
* ----
**/
fun sqrt(number: Number): Number = native("system::SqrtNumberFunctionValue")

/**
* Returns the absolute value of an input number.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to apply the operation to.
* |===
*
* == Example
*
* This example returns the absolute value of the input numbers.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: abs(-2),
*   b: abs(2.5),
*   c: abs(-3.4),
*   d: abs(3)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 2,
*   "b": 2.5,
*   "c": 3.4,
*   "d": 3
* }
* ----
**/
fun abs(number: Number): Number = native("system::AbsNumberFunctionValue")

/**
* Rounds an input number up to the nearest whole number.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to round.
* |===
*
* == Example
*
* This example rounds numbers up to the nearest whole numbers. Notice that `2.1`
* rounds up to `3`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
* {
*   a: ceil(1.5),
*   b: ceil(2.1),
*   c: ceil(3)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 2,
*   "b": 3,
*   "c": 3
* }
* ----
**/
fun ceil(number: Number): Number = native("system::CeilNumberFunctionValue")

/**
* Rounds an input number down to the nearest whole number.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to apply the operation to.
* |===
*
* == Example
*
* This example rounds numbers down to the nearest whole numbers. Notice that
* `1.5` rounds down to `1`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "a" : floor(1.5),
*   "b" : floor(2.2),
*   "c" : floor(3)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 1,
*   "b": 2,
*   "c": 3
* }
* ----
**/
fun floor(number: Number): Number = native("system::FloorNumberFunctionValue")

/**
* Returns the type of an input value.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | A string, object, array, number, or other supported type.
* |===
*
* == Example
*
* This example identifies the type of the input string.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* typeOf("A Text")
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* "String"
* ----
*/
fun typeOf <T>(value: T): Type<T> = native("system::TypeOfAnyFunctionValue")

/**
* Rounds an input number up or down to the nearest whole number.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to round.
* |===
*
* == Example
*
* This example rounds decimal numbers to the nearest whole numbers.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   a: round(1.2),
*   b: round(4.6),
*   c: round(3.5)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": 1,
*   "b": 5,
*   "c": 4
* }
* ----
**/
fun round(number: Number): Number = native("system::RoundNumberFunctionValue")

/**
* Returns `true` if the given list (array) is empty, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `elements` | The list (an array) to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input array is empty.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "empty" : isEmpty([]),
*   "nonEmpty" : isEmpty([1])
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "empty": true,
*    "nonEmpty": false
* }
* ----
*/
fun isEmpty(elements: Array<Any>): Boolean = native("system::EmptyArrayFunctionValue")

/**
* Returns `true` if the given string is empty, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | A string to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input strings are empty.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "empty" : isEmpty(""),
*   "nonEmpty" : isEmpty("DataWeave")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "empty": true,
*     "nonEmpty": false
*   }
* ----
*/
fun isEmpty(value: String): Boolean = native("system::EmptyStringFunctionValue")

/**
* Returns `true` if the input is `null`.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | `Null` is the value in this case.
* |===
*
* == Example
*
* This example indicates whether the input is `null`.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "null" : isEmpty(null)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "null": true
* }
* ----
*/
fun isEmpty(value: Null): Boolean = true

/**
* Returns `true` if the given object is empty, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `value` | The object to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input object is empty.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "empty" : isEmpty({}),
*   "nonEmpty" : isEmpty({name: "DataWeave"})
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "empty": true,
*     "nonEmpty": false
*   }
* ----
*/
fun isEmpty(value: Object): Boolean = native("system::EmptyObjectFunctionValue")

/**
* Returns `true` if it receives a `DateTime` for a leap year, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `dateTime` | The date-time (`DateTime` type) to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input is a leap year.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "leapYear" : isLeapYear(|2016-10-01T23:57:59Z|),
*   "leapYear" : isLeapYear(|2016-10-01T23:57:59-03:00|),
*   "leapYear": isLeapYear(|2017-10-01T23:57:59Z|),
*   "leapYear": isLeapYear(|2017-10-01T23:57:59-03:00|)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "leapYear": true,
*     "leapYear": true,
*     "leapYear": false,
*     "leapYear": false
*   }
* ----
*/
fun isLeapYear(dateTime: DateTime): Boolean = native("system::LeapDateTimeFunctionValue")

/**
* Returns `true` if the input `Date` is a leap year, 'false' if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `date` | The date (`Date` type) to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input is a leap year.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "leapYear" : isLeapYear(|2016-10-01|),
*   "leapYear": isLeapYear(|2017-10-01|)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "leapYear": true,
*     "leapYear": false
*   }
* ----
*/
fun isLeapYear(date: Date): Boolean = native("system::LeapLocalDateFunctionValue")

/**
* Returns `true` if the input local date-time is a leap year, 'false' if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `datetime` | A local date-time (`LocalDateTime` type) to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input is a leap year.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "leapYear" : isLeapYear(|2016-10-01T23:57:59|),
*   "leapYear": isLeapYear(|2017-10-01T23:57:59|)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "leapYear": true,
*     "leapYear": false
*   }
* ----
*/
fun isLeapYear(datetime: LocalDateTime): Boolean = native("system::LeapLocalDateTimeFunctionValue")

/**
* Returns `true` if the given number contains a decimal, `false` if not.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | A number to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input number has a decimal.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "decimal" : isDecimal(1.1),
*   "decimal" : isDecimal(1)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*   {
*     "decimal": true,
*     "decimal": false
*   }
* ----
*/
fun isDecimal(number: Number): Boolean = native("system::DecimalNumberFunctionValue")

/**
* Returns `true` if the given number is an integer (which lacks decimals),
* `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | The number to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input is an integer.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*    "integer" : isInteger(1.1),
*    "integer" : isInteger(1)
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*    "integer": false,
*    "integer": true
* }
* ----
*/
fun isInteger(number: Number): Boolean = native("system::IntegerNumberFunctionValue")

/**
* Returns `true` the given string is empty or completely whitespace, `false`
* if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | A string to evaluate.
* |===
*
* == Example
*
* This example indicates whether the given values are blank.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output  application/json
* ---
* {
*   "empty" : isBlank(""),
*   "withSpaces" : isBlank("      "),
*   "withText" : isBlank(" 1223")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "empty": true,
*   "withSpaces": true,
*   "withText": false
* }
* ----
*/
fun isBlank(text: String | Null): Boolean = isEmpty(trim(text))

/**
* Returns `true` if the given number is odd, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | A number (`Number` type) to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input numbers are odd.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output  application/json
* ---
* {
*    "isOdd" :
*    [ isOdd(0), isOdd(1), isOdd(2) ]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "isOdd": [
*     false,
*     true,
*     false
*   ]
* }
* ----
*/
fun isOdd(number: Number): Boolean =  mod(number, 2) != 0

/**
* Returns `true` if the given number is even, `false` if not.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `number` | A number to evaluate.
* |===
*
* == Example
*
* This example indicates whether the input numbers are even.
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output  application/json
* ---
* {
*    "isEven" :
*    [ isEven(0), isEven(1), isEven(2)]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*   "isEven" : [
*     true,
*     false,
*     true
*   ]
* }
* ----
*/
fun isEven(number: Number): Boolean =
  mod(number, 2) == 0

/**
* Returns the lowest value of comparable elements in a list (an array).
*
*
* Returns null when list is empty. Returns an error if the items in the array
* are not comparable.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `item` | Element in the list (of type `Number`, `Boolean`, `DateTime`, `LocalDateTime`, `Date`, `LocalTime`, `Time`, or `TimeZone`). Can be referenced with `&#36;`.
* |===
*
* == Example
*
* This example returns the lowest value within objects (key-value pairs) in
* an array. Notice that it uses `item.a` to access the value of the object. You
* can also write the expression in the source like this:
* `[ { "a" : 1 }, { "a" : 3 }, { "a" : 2 } ] minBy &#36;`
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output  application/json
* ---
* [ { "a" : 1 }, { "a" : 2 }, { "a" : 3 } ] minBy (item) -> item.a
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "a" : 1 }
* ----
*/
fun minBy<T>(array: Array<T>, criteria: (item: T) -> Comparable): T | Null =
  reduce(array, (val, prev) ->
    if(criteria(val) < criteria(prev))
      val
    else
      prev
  )

/**
* Returns the highest value of comparable elements in the given list (an array).
*
*
* Returns null when the list is empty. Returns an error if the items in the list
* are not comparable.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `item` | Element in the given array (of type `Number`, `Boolean`, `DateTime`, `LocalDateTime`, `Date`, `LocalTime`, `Time`, or `TimeZone`). Can be referenced with `&#36;`.
* |===
*
* == Example
*
* This example returns the highest value within objects (key-value pairs) in
* an array. Notice that it uses `item.a` to access the value of the object. You
* can also write the expression in the source like this:
* `[ { "a" : 1 }, { "a" : 3 }, { "a" : 2 } ] maxBy &#36;.a`
*
* === Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output  application/json
* ---
* [ { "a" : 1 }, { "a" : 3 }, { "a" : 2 } ] maxBy ((item) -> item.a)
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* { "a" : 3 }
* ----
*/
fun maxBy<T>(array: Array<T>, criteria: (item: T) -> Comparable): T | Null =
  reduce(array, (val, prev) ->
    if (criteria(val) > criteria(prev))
      val
    else
      prev
  )

/**
* Returns the number of days between two dates.
*
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `from` | From date (a `Date` type).
* | `to` | To date (a `Date` type).
* |===
*
* == Example
*
* This example returns the number of days between the specified dates.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* output application/json
* ---
* {
*   "days" : daysBetween("2016-10-01T23:57:59-03:00",
*   "2017-10-01T23:57:59-03:00")
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
*  {
*    "days" : 365
*  }
* ----
*/
fun daysBetween(from: Date, to: Date): Number = native("system::daysBetween")
