/**
* This module contains helper functions for working with arrays.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Arrays` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Returns `true` if an element in the list (an array) matches the specified condition.
 *
 *
 * The function stops iterating after the first match to an element in the list.
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `list` | The input list.
 * | `condition` | A condition (or expression) to apply to the input list.
 * |===
 *
 * == Example
 *
 * This example applies a variety of expressions to the input lists.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 *  %dw 2.0
 *  import * from dw::core::Arrays
 *  output application/json
 *  ---
 *  { "results" : [
 *      "ok" : [
 *        [1,2,3] some (($ mod 2) == 0),
 *        [1,2,3] some (($ mod 2) == 1),
 *        [1,2,3,4,5,6,7,8] some (log('should stop at 2 ==', $) == 2),
 *        [1,2,3] some ($ == 1),
 *        [1,1,1] some ($ == 1),
 *        [1] some ($ == 1)
 *      ],
 *      "err" : [
 *        [1,2,3] some ($ == 100),
 *        [1] some ($ == 2)
 *      ]
 *    ]
 * }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    "results": [
 *      {
 *        "ok": [ true, true, true, true, true, true ]
 *      },
 *      {
 *        "err": [ false, false ]
 *      }
 *    ]
 *  }
 * ----
 */
fun some<T>(list: Array<T>, condition: (T) -> Boolean): Boolean =
  list match {
    case [] -> false
    case [head ~ tail] ->
      if(condition(head))
        true
      else
        some(tail, condition)
  }

/**
 * Returns `true` if every element in the list (array) matches the condition.
 *
 *
 * The function stops iterating after the first negative evaluation of a list.
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `list` | The input list.
 * | `condition` | A condition (or expression) to apply to the input list.
 * |===
 *
 * == Example
 *
 * This example applies a variety of expressions to the input lists.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Arrays
 * var arr0 = [] as Array<Number>
 * output application/json
 * ---
 * { "results" : [
 *      "ok" : [
 *         [1,1,1] every ($ == 1),
 *         [1] every ($ == 1)
 *      ],
 *      "err" : [
 *         [1,2,3] every ((log('should stop at 2 ==', $) mod 2) == 1),
 *         [1,1,0] every ($ == 1),
 *         [0,1,1,0] every (log('should stop at 0 ==', $) == 1),
 *         [1,2,3] every ($ == 1),
 *         arr0 every true,
 *      ]
 *    ]
 *  }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    "results": [
 *      {
 *        "ok": [ true, true ]
 *      },
 *      {
 *        "err": [ false, false, false, false, false ]
 *      }
 *    ]
 *  }
 * ----
 */
fun every<T>(list: Array<T>, condition: (T) -> Boolean): Boolean = do {
  fun private_every<T>(list: Array<T>, condition: (T) -> Boolean): Boolean = do {
    list match {
      case [] -> true
      case [head ~ tail] ->
        if(condition(head))
          private_every(tail, condition)
        else
          false
    }
  }
  ---
  list match {
    case [] -> false
    else -> private_every(list, condition)
  }
}

/**
 * Counts the elements in a list (array) that match the results of a function.
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `array` | The input list that contains elements to match.
 * | `matchingFunction` | A function to apply to the input list.
 * |===
 *
 * == Example
 *
 * This counts the values in the list that are equal to the result of the
 * `matchingFunction` (`(($ mod 2) == 0)`).
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 *  %dw 2.0
 *  import * from dw::core::Arrays
 *  output application/json
 *  ---
 * { "countBy" : [1, 2, 3, 4] countBy (($ mod 2) == 0) }
 * ----
 *
 * === Output
 *
 * [source,JSON,linenums]
 * ----
 * { "countBy": 2 }
 * ----
 */
fun countBy<T>(array: Array<T>, matchingFunction: (T) -> Boolean): Number =
  (array reduce (item: T, carry: Number = 0) ->
    if(matchingFunction(item))
      carry + 1
    else
      carry) default 0

/**
 * Returns the sum of the values of the elements in a list (an array).
 *
 *
 * == Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `array` | The input list.
 * | `numberSelector` | A DataWeave selector that selects the values of the numbers in the input list.
 * |===
 *
 * == Example
 *
 * This example calculates the sum of the values of elements some lists.  Notice
 * that both of the `sumBy` function calls produce the same result.
 *
 * === Source
 *
 * [source,DataWeave, linenums]
 * ----
 *  %dw 2.0
 *  import * from dw::core::Arrays
 *  output application/json
 *  ---
 * {
 *   "sumBy" : [
 *     [ { a: 1 }, { a: 2 }, { a: 3 } ] sumBy $.a,
 *     sumBy([ { a: 1 }, { a: 2 }, { a: 3 } ], (item) -> item.a)
 *   ]
 * }
 * ----
 *
 * === Output
 *
 * [source,json,linenums]
 * ----
 * { "sumBy" : [ 6, 6 ] }
 * ----
 */
fun sumBy<T>(array: Array<T>, numberSelector: (T) -> Number): Number =
  (array reduce (item: T, carry: Number = 0) ->
    numberSelector(item) + carry) default 0

/**
* Breaks up a list (array) into sub-lists (sub-arrays) that contain the
* specified number of elements.
*
*
* If there are fewer elements in the input array than the specified number,
* the function will fill the sub-array with those elements. If there are more
* elements, the function will fill as many sub-arrays needed with the extra
* elements.
*
* == Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `items` | Items in the input array.
* | `amount` | The number of elements allowed per sub-array.
* |===
*
* == Example
*
* This example breaks up arrays into sub-arrays based on the specified `amount`.
*
* === Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::core::Arrays
* output application/json
* ---
* {
*   "divideBy" : [
*       { "divideBy2" : [1, 2, 3, 4, 5] divideBy 2 },
*       { "divideBy2" : [1, 2, 3, 4, 5, 6] divideBy 2 },
*       { "divideBy3" : [1, 2, 3, 4, 5] divideBy 3 }
*   ]
* }
* ----
*
* === Output
*
* [source,JSON,linenums]
* ----
* {
*  "divideBy": [
*   {
*     "divideBy2": [
*       [ 1, 2 ],
*       [ 3, 4 ],
*       [ 5 ]
*     ]
*   },
*   {
*     "divideBy2": [
*       [ 1, 2 ],
*       [ 3, 4 ],
*       [ 5, 6 ]
*     ]
*   },
*     {
*       "divideBy3": [
*         [ 1, 2, 3 ],
*         [ 4, 5 ]
*       ]
*     }
*  ]
* }
* ----
*/
fun divideBy<T>(items: Array<T>, amount: Number): Array<Array<T>> = do {
    fun internalDivideBy<T>(items: Array<T>, amount: Number, carry:Array<T> ): Array<Array<T>> =
      items match {
          case [x ~ xs] ->
            if(sizeOf(carry) == amount - 1)
                [carry << x ~ internalDivideBy(xs, amount, [])]
            else
               internalDivideBy(xs, amount, carry << x )
          else ->
            if(isEmpty(carry))
             []
            else
             [carry]
      }
    ---
    internalDivideBy(items, amount, [])
}
