/**
* This module handles value modification in a simple and nice way.
*
*
* *Since Version: 2.2.2*
*/
%dw 2.0

import * from dw::util::Tree

/**
* This functions creates a `PathElement` that selects an *Attribute*.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | namespace? | The namespace of the attribute to select. If not specified null value is set
* | name | The name of the attribute to select.
* |===
*
* === Example
*
* This shows how to create an attr with a given namespace.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::util::Values
* ns ns0 http://acme.com/fo
* ---
* attr(ns0 , "myAttr")
* ----
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "kind": "Attribute",
*    "namespace": "http://acme.com/foo",
*    "selector": "myAttr"
*  }
* ----
**/
fun attr(namespace: Namespace | Null = null, name: String): PathElement =
  {
    kind: ATTRIBUTE_TYPE,
    namespace: namespace,
    selector: name
  }

/**
*
* This functions creates a `PathElement` that selects an *Object field*.
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | namespace? | The namespace of the field to select. If not specified null value is set
* | name | The name of the attribute to select.
* |===
*
* === Example
*
* This example shows how the `field` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*  %dw 2.0
*  output application/json
*  import * from dw::util::Values
*  ns ns0 http://acme.com/foo
*  ---
*  field(ns0 , "myAttr")
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "kind": "Object",
*    "namespace": "http://acme.com/foo",
*    "selector": "myAttr"
*  }
* ----
**/
fun field(namespace: Namespace | Null = null, name: String): PathElement =
  {
    kind: OBJECT_TYPE,
    namespace: namespace,
    selector: name
  }

/**
* This functions creates a `PathElement` that selects an *Array element*.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | index |
* |===
*
* === Example
*
* This example shows how the `index` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
*  %dw 2.0
*  output application/json
*  import * from dw::util::Values
*  ns ns0 http://acme.com/foo
*  ---
*  index(0)
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "kind": "Array",
*    "namespace": null,
*    "selector": 0
*  }
* ----
**/
fun index(index: Number): PathElement =
  {
    kind: ARRAY_TYPE,
    namespace: null,
    selector: index
  }


/**
* Helper function to make mask work with null correctly
**/
fun mask(value: Null, fieldName: String | Number | PathElement): Any =
    (newValueProvider: (oldValue: Any, path: Path) -> Any): Any -> null

/**
* The *mask* function allows to replace all *simple* elements that matches the specified criteria.
* Simple elements are the ones that don't have child elements, so when the value are not Object or Arrays.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value that is going to be mask
* | selector | The PathElement selector
* |===
*
* === Example
*
* This example shows how to mask
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::util::Values
* ns ns0 http://acme.com/foo
* ---
* [{name: "Peter Parker", password: "spiderman"}, {name: "Bruce Wayne", password: "batman"}] mask field("password") with "*****"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* [
*    {
*      "name": "Peter Parker",
*      "password": "*****"
*    },
*    {
*      "name": "Bruce Wayne",
*      "password": "*****"
*    }
*  ]
* ----
**/
fun mask(value: Any, selector: PathElement): Any =
  (newValueProvider: (oldValue: Any, path: Path) -> Any): Any -> do {
      mapLeafValues(value, (v, p) -> do {
          var lastSegment = p[-1]
          ---
          if (selector.kind == lastSegment.kind and lastSegment.selector == selector.selector and (selector.namespace == null or lastSegment.namespace == selector.namespace))
            newValueProvider(v, p)
          else
            v
        })
    }

/**
* This overload of *mask* allows to select a field by its name.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value that is going to be mask
* | fieldName | The name of the field to mask
* |===
*
* === Example
*
* This example shows how to mask
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::util::Values
* ns ns0 http://acme.com/foo
* ---
* [{name: "Peter Parker", password: "spiderman"}, {name: "Bruce Wayne", password: "batman"}] mask "password" with "*****"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* [
*    {
*      "name": "Peter Parker",
*      "password": "*****"
*    },
*    {
*      "name": "Bruce Wayne",
*      "password": "*****"
*    }
*  ]
* ----
**/
fun mask(value: Any, fieldName: String): Any =
  mask(value, field(fieldName))

/**
*
* This overload of *mask* allows to select an Array by its index
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value | The value to mask
* | i | The index to mask
* |===
*
* === Example
*
* This example shows how the `mask` behaves all the first elements of all the arrays
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* [[123, true], [456, true]] mask 1 with false
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* [
*    [
*      123,
*      false
*    ],
*    [
*      456,
*      false
*    ]
*  ]
* ----
**/

fun mask(value: Any, i: Number): Any =
  mask(value, index(i))



/**
*  Updates an *Object* field with the specified value. This function will return a new *Object* with the value of the specified field changed.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The object to be update
* | fieldName | The field name
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* {name: "Mariano"} update "name" with "Data Weave"
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* {
*   "name": "Data Weave"
* }
* ----
**/
fun update(objectValue: Object, fieldName: String):Any =
  update(objectValue, field(fieldName))

/**
*  Updates an Object field with the specified value. This function will return a new *Object* with the value of the specified field changed.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The object to be update
* | fieldName | The field name
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* {name: "Mariano"} update field("name") with "Data Weave"
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* {
*   "name": "Data Weave"
* }
* ----
**/
fun update(objectValue: Object, fieldName: PathElement):Any =
  (newValueProvider: (oldValue: Any, index: Number) -> Any): Object -> do {
      if (objectValue[fieldName.selector]?)
        objectValue mapObject ((value, key, index) -> if (key ~= fieldName.selector and (fieldName.namespace == null or fieldName.namespace == key.#))
            {
              (key): newValueProvider(value, index)
            }
          else
            {
              (key): value
            })
      else
        objectValue
    }

/**
*  Updates an Array index with the specified value. This function will return a new *Array* with the value of the specified index changed.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The Array to be update
* | indexToUpdate | The index of the array to update
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* [1,2,3] update 1 with 5
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* [
*    1,
*    5,
*    3
*  ]
* ----
**/
fun update(arrayValue: Array, indexToUpdate: Number):Any =
  update(arrayValue, index(indexToUpdate))


/**
*  Updates all Objects inside the specified Array with the given value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The Array of Object to be update
* | indexToUpdate | The field name to update
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* [{name: "Mariano"}] update "name" with "Data Weave"
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* [{
*   "name": "Data Weave"
* }]
* ----
**/
fun update(arrayValue: Array, indexToUpdate: String):Any =
  update(arrayValue, field(indexToUpdate))

/**
*  Updates an Array index with the specified value. This function will return a new *Array* with the value of the specified index changed.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The Array to be update
* | indexToUpdate | The index of the array to update
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* [1,2,3] update index(1) with 5
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* [
*    1,
*    5,
*    3
*  ]
* ----
**/
fun update(arrayValue: Array, indexToUpdate: PathElement): Any =
      (newValueProvider: (oldValue: Any, index: Number) -> Any): Any -> do {
          if(indexToUpdate.kind != ARRAY_TYPE)
                arrayValue map ((item, index) -> item update indexToUpdate with newValueProvider($,$$))
          else
              if (arrayValue[indexToUpdate.selector as Number]?)
                arrayValue map ((value, index) ->
                    if (index == indexToUpdate.selector)
                        newValueProvider(value, index)
                    else
                        value)
              else
                arrayValue
     }

/**
*
* Updates the value at the specified path with the given value
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | objectValue | The value to be update
* | path | The path to update
* |===
*
* === Example
*
* This example update the name of the user
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::util::Values
* output application/json
* ---
* {user: {name: "Mariano"}} update ["user", field("name")] with "Data Weave"
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* {
*    "user": {
*      "name": "Data Weave"
*    }
*  }
* ----
**/
fun update(value: Array | Object | Null, path: Array<String | Number | PathElement>):Any = do {
    fun updateAttr(value: Null, path: Array<String | Number | PathElement>):Any =
      (newValueProvider: (oldValue: Any, index: Number) -> Any) -> null

    fun updateAttr(value: Array | Object | Null, path: Array<String | Number | PathElement>):Any = do {
      (newValueProvider: (oldValue: Any, index: Number) -> Any) -> do {

          fun doUpdateAttr(objectValue: Object, f: String | Number, attr: String | Number | PathElement) = do {
            var fieldName = f as String
            ---
            if (objectValue[fieldName]?)
              objectValue mapObject ((value, key, index) -> if (key ~= fieldName)
                  {
                    (key) @((update(key.@, attr) with newValueProvider($, $$))): value
                  }
                else
                  {
                    (key): value
                  })
            else
              objectValue
          }

          fun doUpdateAttr(objectValue: Object, f: PathElement, attr: String | Number | PathElement) = do {
            var fieldName = f.selector as String
            ---
            if (objectValue[fieldName]?)
              objectValue mapObject ((value, key, index) -> if (key ~= fieldName and (f.namespace == null or f.namespace == key.#))
                  {
                    (key) @((update(key.@, attr) with newValueProvider($, $$))): value
                  }
                else
                  {
                    (key): value
                  })
            else
              objectValue
          }

          fun doUpdateAttr(objectValue: Array | Null, fieldName: String | Number | PathElement, attr: String | Number | PathElement) =
            objectValue
          ---
          if (sizeOf(path) > 2)
            value update path[0 to -3] with doUpdateAttr($, path[-2], path[-1])
          else
            doUpdateAttr(value, path[-2], path[-1])
        }
    }
    ---
    if (path[-1] is PathElement and (path[-1] as PathElement).kind == ATTRIBUTE_TYPE)
        updateAttr(value, path)
    else
        (newValueProvider: (oldValue: Any, index: Number) -> Any): Any -> do {

            fun doRecUpdate(value: Array | Object, path: Array<String | Number | PathElement>): Any =
              path match {
                case [x ~ xs] -> if (isEmpty(xs))
                  value update x with newValueProvider($, $$)
                else
                  value update x with doRecUpdate($, xs)
                case [] -> value
              }

            fun doRecUpdate(value: Any, path: Array<String | Number | PathElement>): Any =
              value
            ---
            doRecUpdate(value, path)
          }
}

/**
*
* Helper function to make update null friendly.
**/
fun update(value: Null, toUpdate: Number | String | PathElement):Any =
  (newValueProvider: (oldValue: Any, index: Number) -> Any) -> null



