/**
* This module allows to introspect a Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*/
%dw 2.0

@Since(version = "2.3.0")
type QName = {
    localName: String,
    namespace: Namespace | Null
}

@Since(version = "2.3.0")
type Attribute = {
    name: QName,
    required: Boolean,
    value: Type
}

@Since(version = "2.3.0")
type Field = {
    key: {
        name: QName,
        attributes: Array<Attribute>
    },
    required: Boolean,
    repeated: Boolean,
    value: Type
}

@Since(version = "2.3.0")
type FunctionParam = {
   paramType: Type,
   optional: Boolean
}

/**
* Returns true if a Type represent a Boolean Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isBooleanType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ABoolean = Boolean
* output application/json
* ---
* {
*    a: isBooleanType(ABoolean),
*    b: isBooleanType(Boolean),
*    c: isBooleanType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isBooleanType(t:Type): Boolean = baseTypeOf(t) == Boolean

/**
* Returns true if a Type represent a Number Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isNumberType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ANumber = Number
* output application/json
* ---
* {
*    a: isNumberType(ANumber),
*    b: isNumberType(Number),
*    c: isNumberType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNumberType(t:Type): Boolean  = baseTypeOf(t) == Number

/**
* Returns true if a Type represent a Date Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isDateType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ADate = Date
* output application/json
* ---
* {
*    a: isDateType(ADate),
*    b: isDateType(Date),
*    c: isDateType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isDateType(t:Type): Boolean  = baseTypeOf(t) == Date

/**
* Returns true if a Type represent a LocalDateTime Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isDateTimeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ADateTime = DateTime
* output application/json
* ---
* {
*    a: isDateTimeType(ADateTime),
*    b: isDateTimeType(DateTime),
*    c: isDateTimeType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isDateTimeType(t:Type): Boolean  = baseTypeOf(t) == DateTime

/**
* Returns true if a Type represent a DateTime Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isLocalDateTimeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ALocalDateTime = LocalDateTime
* output application/json
* ---
* {
*    a: isLocalDateTimeType(ALocalDateTime),
*    b: isLocalDateTimeType(LocalDateTime),
*    c: isLocalDateTimeType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLocalDateTimeType(t:Type): Boolean  = baseTypeOf(t) == LocalDateTime

/**
* Returns true if a Type represent a Time Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isTimeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ATime = Time
* output application/json
* ---
* {
*    a: isTimeType(ATime),
*    b: isTimeType(Time),
*    c: isTimeType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTimeType(t:Type): Boolean  = baseTypeOf(t) == Time

/**
* Returns true if a Type represent a LocalTime Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isLocalTimeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ALocalTime = LocalTime
* output application/json
* ---
* {
*    a: isLocalTimeType(ALocalTime),
*    b: isLocalTimeType(LocalTime),
*    c: isLocalTimeType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLocalTimeType(t:Type): Boolean  = baseTypeOf(t) == LocalTime

/**
* Returns true if a Type represent a TimeZone Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isTimeZoneType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ATimeZone = TimeZone
* output application/json
* ---
* {
*    a: isTimeZoneType(ATimeZone),
*    b: isTimeZoneType(TimeZone),
*    c: isTimeZoneType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTimeZoneType(t:Type): Boolean  = baseTypeOf(t) == TimeZone

/**
* Returns true if a Type represent a Uri Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isUriType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AUri = Uri
* output application/json
* ---
* {
*    a: isUriType(AUri),
*    b: isUriType(Uri),
*    c: isUriType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isUriType(t:Type): Boolean  = baseTypeOf(t) == Uri

/**
* Returns true if a Type represent a Namespace Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isNamespaceType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ANamespace = Namespace
* output application/json
* ---
* {
*    a: isNamespaceType(ANamespace),
*    b: isNamespaceType(Namespace),
*    c: isNamespaceType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNamespaceType(t:Type): Boolean  = baseTypeOf(t) == Namespace


/**
* Returns true if a Type represent a Range Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isRangeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ARange = Range
* output application/json
* ---
* {
*    a: isNamespaceType(ARange),
*    b: isNamespaceType(Range),
*    c: isNamespaceType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isRangeType(t:Type): Boolean  = baseTypeOf(t) == Range


/**
* Returns true if a Type represent a Period Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isPeriodType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type APeriod = Period
* output application/json
* ---
* {
*    a: isPeriodType(APeriod),
*    b: isPeriodType(Period),
*    c: isPeriodType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isPeriodType(t:Type) : Boolean = baseTypeOf(t) == Period


/**
* Returns true if a Type represent a Binary Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isBinaryType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ABinary = Binary
* output application/json
* ---
* {
*    a: isBinaryType(ABinary),
*    b: isBinaryType(Binary),
*    c: isBinaryType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isBinaryType(t:Type): Boolean  = baseTypeOf(t) == Binary


/**
* Returns true if a Type represent a Null Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isNullType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ANull = Null
* output application/json
* ---
* {
*    a: isNullType(ANull),
*    b: isNullType(Null),
*    c: isNullType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNullType(t:Type): Boolean  = baseTypeOf(t) == Null


/**
* Returns true if a Type represent a Nothing Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isNothingType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ANothing = Nothing
* output application/json
* ---
* {
*    a: isNothingType(ANothing),
*    b: isNothingType(Nothing),
*    c: isNothingType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isNothingType(t:Type): Boolean  = baseTypeOf(t) == Nothing


/**
* Returns true if a Type represent a Any Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isAnyType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AAny = Any
* output application/json
* ---
* {
*    a: isAnyType(AAny),
*    b: isAnyType(Any),
*    c: isAnyType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isAnyType(t:Type): Boolean  = baseTypeOf(t) == Any


/**
* Returns true if a Type represent a Type Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isTypeType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = Type
* output application/json
* ---
* {
*    a: isTypeType(AType),
*    b: isTypeType(Type),
*    c: isTypeType(String),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isTypeType(t:Type) : Boolean = baseTypeOf(t) == Type


/**
* Returns true if a Type represent a String Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isStringType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AString = String
* output application/json
* ---
* {
*    a: isStringType(AString),
*    b: isStringType(String),
*    c: isStringType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isStringType(t:Type) : Boolean = baseTypeOf(t) == String


/**
* Returns true if a Type represent a Regex Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isRegexType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ARegex = Regex
* output application/json
* ---
* {
*    a: isRegexType(ARegex),
*    b: isRegexType(Regex),
*    c: isRegexType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isRegexType(t:Type): Boolean  = baseTypeOf(t) == Regex


/**
* Returns true if a Type represent a Key Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isKeyType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AKey = Key
* output application/json
* ---
* {
*    a: isKeyType(AKey),
*    b: isKeyType(Key),
*    c: isKeyType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isKeyType(t:Type): Boolean  = baseTypeOf(t) == Key


/**
* Returns true if a Type represent a literal Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isLiteralType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type ALiteralType = "Mariano"
* output application/json
* ---
* {
*    a: isLiteralType(ALiteralType),
*    b: isLiteralType(Boolean)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isLiteralType(t: Type):Boolean = native("system::IsLiteralTypeFunction")


/**
* Returns true if a Type represent a Function Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isFunctionType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AFunction = (String) -> String
* output application/json
* ---
* {
*    a: isFunctionType(AFunction),
*    b: isFunctionType(Boolean)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isFunctionType(t:Type): Boolean  = native("system::IsFunctionTypeFunction")


/**
* Returns the value of the literal Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `literalValueOf` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = "Mariano"
* output application/json
* ---
* {
*    a: literalValueOf(AType)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "Mariano"
* }
* ----
**/
@Since(version = "2.3.0")
fun literalValueOf(t:Type): String | Number | Boolean = native("system::LiteralTypeValueFunction")



/**
* Returns true if the given type is an Object Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isObjectType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = {name: String}
* output application/json
* ---
* {
*    a: isObjectType(AType),
*    b: isObjectType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isObjectType(t:Type): Boolean = native("system::IsObjectTypeFunction")


/**
* Returns true if the given type is an Array Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isArrayType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = Array<String>
* output application/json
* ---
* {
*    a: isObjectType(AType),
*    b: isObjectType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isArrayType(t:Type) : Boolean = native("system::IsArrayTypeFunction")


/**
* Returns true if the given type is a Union Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isUnionType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = String | Number
* output application/json
* ---
* {
*    a: isUnionType(AType),
*    b: isUnionType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isUnionType(t:Type): Boolean = native("system::IsUnionTypeFunction")



/**
* Returns true if the given type is an Intersection Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `isIntersectionType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = {name: String} & {age: Number}
* output application/json
* ---
* {
*    a: isIntersectionType(AType),
*    b: isIntersectionType(Boolean),
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": true,
*   "b": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isIntersectionType(t:Type):Boolean = native("system::IsIntersectionTypeFunction")

/**
* Returns an Array of all the Types the Union Type is defined with. It will fail if it is not an Union Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `unionItems` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = String | Number
* output application/json
* ---
* {
*    a: unionItems(AType)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": ["String","Number"]
* }
* ----
**/
@Since(version = "2.3.0")
fun unionItems(t:Type): Array<Type>  = native("system::UnionTypeItemsFunction")


/**
*  Returns true if the specified type is a Reference.
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t |
* |===
*
* === Example
*
* This example shows how the `isReferenceType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* import * from dw::core::Types
*  type AArray = Array<String> {n: 1}
*  type AArray2 = Array<AArray>
*
*  ---
*  {
*      a: isReferenceType( AArray),
*      b: isReferenceType(arrayItem(AArray2)),
*      c: isReferenceType(String)
*  }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": false,
*   "b": true,
*   "c": false
* }
* ----
**/
@Since(version = "2.3.0")
fun isReferenceType(t: Type):Boolean = native("system::IsReferenceTypeFunction")

/**
* Returns the name of a given type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The Type to query
* |===
*
* === Example
*
* This example shows how the `nameOf` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* import * from dw::core::Types
* type AArray = Array<String> {n: 1}
* type AArray2 = Array<String>
*
* ---
*  {
*      a: nameOf(AArray),
*      b: nameOf(AArray2),
*      c: nameOf(String)
*  }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": "AArray",
*    "b": "AArray2",
*    "c": "String"
*  }
* ----
**/
@Since(version = "2.3.0")
fun nameOf(t: Type): String = native("system::NameOfFunction")


/**
* Returns an Array of all the Types the Intersection Type is defined with. It will fail if it is not an Intersection Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `intersectionItems` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = {name: String} & {age: Number}
* output application/json
* ---
* {
*    a: intersectionItems(AType)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": ["Object","Object"]
* }
* ----
**/
@Since(version = "2.3.0")
fun intersectionItems(t:Type): Array<Type>  = native("system::IntersectionTypeItemsFunction")

/**
* Returns an the kind of Array the given type is. It will fail if it is not an Array Type.
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `intersectionItems` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = {name: String} & {age: Number}
* output application/json
* ---
* {
*    a: intersectionItems(AType)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": ["Object","Object"]
* }
* ----
**/
@Since(version = "2.3.0")
fun arrayItem(t: Type): Type = native("system::ArrayItemTypeFunction")


/**
* Returns an the base Type the given Type
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `baseTypeOf` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = String {format: "YYYY-MM-dd"}
* output application/json
* ---
* {
*    a: baseTypeOf(AType)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "String"
* }
* ----
**/
@Since(version = "2.3.0")
fun baseTypeOf(t:Type): Type = native("system::BaseTypeFunction")


/**
* Returns an the metadata attached to the given type.
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The type to be checked
* |===
*
* === Example
*
* This example shows how the `metadataOf` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Types
*
* type AType = String {format: "YYYY-MM-dd"}
* output application/json
* ---
* {
*    a: metadataOf(AType)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": {"format": "YYYY-MM-dd"}
* }
* ----
**/
@Since(version = "2.3.0")
fun metadataOf(t:Type) : {} = native("system::MetadataOfFunction")

/**
*  Returns the list of parameters of the given function type. Fails if the provided type is not a Function
*
* _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type
* |===
*
* === Example
*
* This example shows how the `functionParamTypes` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* import * from dw::core::Types
*
*
* type AFunction = (String, Number) -> Number
* type AFunction2 = () -> Number
* ---
* {
*     a: functionParamTypes(AFunction),
*     b: functionParamTypes(AFunction2)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*  {
*     "a": [
*       {
*         "paramType": "String",
*         "optional": false
*       },
*       {
*         "paramType": "Number",
*         "optional": false
*       }
*     ],
*     "b": [
*
*     ]
*   }
* ----
**/
@Since(version = "2.3.0")
fun functionParamTypes(t: Type): Array<FunctionParam> = native("system::FunctionParamTypesFunction")


/**
*  Returns the type of the return type of a Function. Fails if the provided type is not a Function
*
*  _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type
* |===
*
* === Example
*
* This example shows how the `functionReturnType` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* import * from dw::core::Types
*
*
* type AFunction = (String, Number) -> Number
* type AFunction2 = () -> Number
* ---
* {
*     a: functionReturnType(AFunction),
*     b: functionReturnType(AFunction2)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*  {
*     "a": "Number",
*     "b": "Number"
*   }
* ----
**/
@Since(version = "2.3.0")
fun functionReturnType(t: Type): Type | Null = native("system::FunctionReturnTypeFunction")

/**
*  Returns the array of fields of the given object type. Fails if the type is not an Object
*
*  _Introduced in DataWeave 2.3.0. Supported by Mule 4.3 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | t | The function type
* |===
*
* === Example
*
* This example shows how the `objectFields` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::core::Types
*
* ns ns0 http://acme.com
*
* type ADictionary = {_ : String}
* type ASchema = {ns0#name @(ns0#foo: String): {}}
* type AUser = {name @(foo?: String,l: Number)?: String, lastName*: Number}
* ---
* {
*     a: objectFields(ADictionary),
*     b: objectFields(ASchema),
*     c: objectFields(Object),
*     d: objectFields(AUser)
* }
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": [
*     {
*       "key": {
*         "name": {
*           "localName": "_",
*           "namespace": null
*         },
*         "attributes": [
*
*         ]
*       },
*       "required": true,
*       "repeated": false,
*       "value": "String"
*     }
*   ],
*   "b": [
*     {
*       "key": {
*         "name": {
*           "localName": "name",
*           "namespace": "http://acme.com"
*         },
*         "attributes": [
*           {
*             "name": {
*               "localName": "foo",
*               "namespace": "http://acme.com"
*             },
*             "value": "String",
*             "required": true
*           }
*         ]
*       },
*       "required": true,
*       "repeated": false,
*       "value": "Object"
*     }
*   ],
*   "c": [
*
*   ],
*   "d": [
*     {
*       "key": {
*         "name": {
*           "localName": "name",
*           "namespace": null
*         },
*         "attributes": [
*           {
*             "name": {
*               "localName": "foo",
*               "namespace": null
*             },
*             "value": "String",
*             "required": false
*           },
*           {
*             "name": {
*               "localName": "l",
*               "namespace": null
*             },
*             "value": "Number",
*             "required": true
*           }
*         ]
*       },
*       "required": false,
*       "repeated": false,
*       "value": "String"
*     },
*     {
*       "key": {
*         "name": {
*           "localName": "lastName",
*           "namespace": null
*         },
*         "attributes": [
*
*         ]
*       },
*       "required": true,
*       "repeated": true,
*       "value": "Number"
*     }
*   ]
* }
* * ----
**/
@Since(version = "2.3.0")
fun objectFields(t: Type): Array<Field>  = native("system::ObjectTypeFieldsFunction")

