/**
* This module contains functions that allow you to interact with the DataWeave
* engine.
*
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::Runtime` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Throws an exception with the specified message.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `message` | An error message (`String`).
 * |===
 *
 * === Example
 *
 * This example returns a failure message `Data was empty` because the expression
 * `(sizeOf(myVar) &lt;= 0)` is `true`. A shortened version of the error message
 * is shown in the Output section below.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::Runtime
 * var result = []
 * output application/json
 * ---
 * if(sizeOf(result) <= 0) fail('Data was empty') else result
 * ----
 *
 * ==== Output
 *
 * [source,TXT,linenums]
 * ----
 * ERROR 2018-07-29 11:47:44,983 ...
 * *********************************
 * Message               : "Data was empty
 * ...
 * ----
 */
fun fail (message: String = 'Error'): Nothing = native("system::fail")

/**
 * Produces an error with the specified message if the expression in
 * the evaluator returns `true`, otherwise returns the value.
 *
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `value` | The value to return only if the `evaluator` expression is `false`.
 * | `evaluator` | Expression that returns `true` or `false`.
 * |===
 *
 * === Example
 *
 * This example produces a runtime error (instead of a SUCCESS message) because
 * the expression `isEmpty(result)` is `true`. It is `true` because an empty
 * object is passed through variable `result`.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import failIf from dw::Runtime
 * var result = {}
 * output application/json
 * ---
 * { "result" : "SUCCESS" failIf (isEmpty(result)) }
 * ----
 *
 * ==== Output
 *
 * [source,TXT,linenums]
 * ----
 * ERROR 2018-07-29 11:56:39,988 ...
 * **********************************
 * Message               : "Failed
 * ----
 */
fun failIf <T>(value: T, evaluator: (value: T) -> Boolean, message: String = 'Failed'): T =
    if(evaluator(value)) fail(message) else value

/**
* A MimeType represented in a String
*/
type MimeType = String

/**
* *EXPERIMENTAL*
*
* An Reader Input. This is input is going to be used as the source content of the reader created by the specified *mimeType*
*/
type ReaderInput = {
    /**
    * The actual input
    */
    value: Binary,
    /**
    * The encoding to be used
    */
    encoding?: String,
    /**
    * The reader properties to be used to parse this input
    */
    properties?: Dictionary<SimpleType>,
    /**
    * The mimeType of this input.
    */
    mimeType: MimeType
}

/**
* *EXPERIMENTAL*
*
* When the *run* function executes with no problem
*/
type RunSuccess = {
   success: true,
   value: Binary,
   mimeType: MimeType,
   encoding?: String,
   logs : Array<LogEntry>
}

/**
* *EXPERIMENTAL*
*
* When the *eval* function executes with no problem
*/
type EvalSuccess = {
   success: true,
   value: Any,
   logs : Array<LogEntry>
}

/**
* *EXPERIMENTAL*
*
* When the *run* or *eval* function failed.
*/
type ExecutionFailure = {
   success: false,
   message: String,
   kind: String,
   stack?: Array<String>,
   location: Location,
   logs : Array<LogEntry>
}

type Location =  {
      start?: Position,
      end?: Position,
      content: String
}

type Position = {
    index: Number,
    line: Number,
    column: Number
}

type LogLevel = "INFO" | "ERROR" | "WARN"

type LogEntry = {
    level: LogLevel,
    timestamp: String,
    message: String
}


type DataFormatProperty = {
    name: String,
    optional: Boolean,
    defaultValue?: Any,
    description: String,
    possibleValues: Array<Any>
}

/**
* *EXPERIMENTAL*
*
* Describes a DataFormat providing all the metadata information.
*/
type DataFormatDescriptor = {
    name: String,
    binary: Boolean,
    defaultEncoding?: String,
    extensions: Array<String>,
    defaultMimeType: String,
    acceptedMimeTypes: Array<String>,
    readerProperties: Array<DataFormatProperty>,
    writerProperties: Array<DataFormatProperty>
}

/**
* *EXPERIMENTAL*
*
* This function is going to be called every time some privilege needs to be granted to the current execution.
* The *grant* is the name of the Privilege i.e: *Resource*
* The *args* represents the list of parameters the function that is requesting the Privilege is being called.
*/
type SecurityManager = (grant: String, args: Array<Any>) -> Boolean

/**
* *EXPERIMENTAL*
*
* This service handles all the logging.
*/
type LoggerService = {
    /**
    * This function is going to be called when the execution starts.
    * The result is going to be sent in every log so for example if some logging header needs to be sent it could be done at initialize and then recover in each log.
    */
    initialize?: () -> {},
    /**
    * This log method is going to be called on every log message
    */
    log: (level: LogLevel, msg: String, context: {}) -> Any,
    /**
    * When the execution finished it is going to be called. Good time to flush any buffer or to gracefully logout.
    */
    shutdown?: () -> Boolean
}

/**
* *EXPERIMENTAL*
*
* Configures the runtime execution with some advance parameters.
*/
type RuntimeExecutionConfiguration = {
    /**
    * The max time this script should take.
    */
    timeOut?: Number,
    /**
    * The default output mimeType if it is not specified in the script
    */
    outputMimeType?: MimeType,
    /**
    * The writer properties to be used when the specified the *outputMimeType* property.
    */
    writerProperties?: Dictionary<SimpleType>,
    /**
    * When an the execution fails what it should happen. When *Handle* (default value) is specify the execution will return *ExecutionFailure* if *Fail* is specify
    * then the *Exception* is going to be propagated.
    */
    onException?: "HANDLE" | "FAIL",
    /**
    * The *SecurityManager* to use in this execution. This security manager is going to be composed with the *current* SecurityManager.
    */
    securityManager?: SecurityManager,
    /**
    * The *LoggerService* to be used in this execution.
    */
    loggerService?: LoggerService

}

/**
*
* *EXPERIMENTAL*
*
* Runs a given script under the provided context. This function will execute this given script in the current runtime.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | fileToExecute | The file name to execute
* | fs | The file system where to look for the files
* | readerInputs | The inputs that are going to be read and bind to the execution
* | inputValues | The inputs that are going to be bind directly to the execution
* | configuration | The runtime configuration.
* |===
*
* === Example
*
* This example shows how the `run` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::Runtime
* var jsonValue = {
*   value: '{"name": "Mariano"}' as Binary {encoding: "UTF-8"},
*   encoding: "UTF-8",
*   properties: {},
*   mimeType: "application/json"
* }
*
* var jsonValue2 = {
*   value: '{"name": "Mariano", "lastName": "achaval"}' as Binary {encoding: "UTF-8"},
*   encoding: "UTF-8",
*   properties: {},
*   mimeType: "application/json"
* }
*
* var invalidJsonValue = {
*   value: '{"name": "Mariano' as Binary {encoding: "UTF-8"},
*   encoding: "UTF-8",
*   properties: {},
*   mimeType: "application/json"
* }
*
*
*
*
* var Utils = "fun sum(a,b) = a +b"
* ---
* {
*   "execute_ok" : run("main.dwl", {"main.dwl": "{a: 1}"}, {"payload": jsonValue }),
*   "logs" : do {
*     var execResult = run("main.dwl", {"main.dwl": "{a: log(1)}"}, {"payload": jsonValue })
*     ---
*     {
*         m: execResult.logs.message,
*         l: execResult.logs.level
*     }
*   },
*   "grant" : run("main.dwl", {"main.dwl": "{a: readUrl(`http://google.com`)}"}, {"payload": jsonValue }, { securityManager: (grant, args) -> false }),
*   "library" : run("main.dwl", {"main.dwl": "Utils::sum(1,2)", "/Utils.dwl": Utils }, {"payload": jsonValue }),
*   "timeout" : run("main.dwl", {"main.dwl": "(1 to 1000000000000) map \$ + 1" }, {"payload": jsonValue }, {timeOut: 2}).success,
*   "execFail" : run("main.dwl", {"main.dwl": "dw::Runtime::fail('My Bad')" }, {"payload": jsonValue }),
*   "parseFail" : run("main.dwl", {"main.dwl": "(1 + " }, {"payload": jsonValue }),
*   "writerFail" : run("main.dwl", {"main.dwl": "output application/xml --- 2" }, {"payload": jsonValue }),
*   "readerFail" : run("main.dwl", {"main.dwl": "output application/xml --- payload" }, {"payload": invalidJsonValue }),
*   "defaultOutput" : run("main.dwl", {"main.dwl": "payload" }, {"payload": jsonValue2}, {outputMimeType: "application/csv", writerProperties: {"separator": "|"}}),
* }
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "execute_ok": {
*     "success": true,
*     "value": "{\n  a: 1\n}",
*     "mimeType": "application/dw",
*     "encoding": "UTF-8",
*     "logs": [
*
*     ]
*   },
*   "logs": {
*     "m": [
*       "1"
*     ],
*     "l": [
*       "INFO"
*     ]
*   },
*   "grant": {
*     "success": false,
*     "message": "The given required permissions: `Resource` are not being granted for this execution.\nTrace:\n  at readUrl (Unknown)\n  at main::main (line: 1, column: 5)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "end": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "content": "Unknown location"
*     },
*     "stack": [
*       "readUrl (anonymous:0:0)",
*       "main (main:1:5)"
*     ],
*     "logs": [
*
*     ]
*   },
*   "library": {
*     "success": true,
*     "value": "3",
*     "mimeType": "application/dw",
*     "encoding": "UTF-8",
*     "logs": [
*
*     ]
*   },
*   "timeout": false,
*   "execFail": {
*     "success": false,
*     "message": "My Bad\nTrace:\n  at fail (Unknown)\n  at main::main (line: 1, column: 1)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "end": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "content": "Unknown location"
*     },
*     "stack": [
*       "fail (anonymous:0:0)",
*       "main (main:1:1)"
*     ],
*     "logs": [
*
*     ]
*   },
*   "parseFail": {
*     "success": false,
*     "message": "Invalid input \"1 + \", expected parameter or parenEnd (line 1, column 2):\n\n\n1| (1 + \n    ^^^^\nLocation:\nmain (line: 1, column:2)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 1,
*         "column": 2
*       },
*       "end": {
*         "index": 4,
*         "line": 1,
*         "column": 6
*       },
*       "content": "\n1| (1 + \n    ^^^^"
*     },
*     "logs": [
*
*     ]
*   },
*   "writerFail": {
*     "success": false,
*     "message": "Trying to output non-whitespace characters outside main element tree (in prolog or epilog), while writing Xml at .",
*     "location": {
*       "content": ""
*     },
*     "stack": [
*
*     ],
*     "logs": [
*
*     ]
*   },
*   "readerFail": {
*     "success": false,
*     "message": "Unexpected end-of-input at payload@[1:18] (line:column), expected '\"', while reading `payload` as Json.\n \n1| {\"name\": \"Mariano\n                    ^",
*     "location": {
*       "content": "\n1| {\"name\": \"Mariano\n                    ^"
*     },
*     "stack": [
*
*     ],
*     "logs": [
*
*     ]
*   },
*   "defaultOutput": {
*     "success": true,
*     "value": "name|lastName\nMariano|achaval\n",
*     "mimeType": "application/csv",
*     "encoding": "UTF-8",
*     "logs": [
*
*     ]
*   }
* }
* ----
**/
@Experimental()
@RuntimePrivilege(requires = "Execution")
fun run(fileToExecute: String, fs: Dictionary<String>, readerInputs: Dictionary<ReaderInput> = {}, inputValues: Dictionary<Any> = {}, configuration: RuntimeExecutionConfiguration = {}): RunSuccess | ExecutionFailure =
      native("system::RunScriptFunctionValue")

/**
* *EXPERIMENTAL*
*
* Evaluates a given script with the specified context and returns the result of that evaluation
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | fileToExecute | The name of the file to execute
* | fs | An Object with the files
* | readerInputs | The reader inputs to bind
* | inputValues | Additional literal values to bind
* | configuration | Runtime configuration
* |===
*
* === Example
*
* This example shows how the `eval` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
*
*  import * from dw::Runtime
*
*  var jsonValue = {
*    value: '{"name": "Mariano"}' as Binary {encoding: "UTF-8"},
*    encoding: "UTF-8",
*    properties: {},
*    mimeType: "application/json"
*  }
*
*  var jsonValue2 = {
*    value: '{"name": "Mariano", "lastName": "achaval"}' as Binary {encoding: "UTF-8"},
*    encoding: "UTF-8",
*    properties: {},
*    mimeType: "application/json"
*  }
*
*  var invalidJsonValue = {
*    value: '{"name": "Mariano' as Binary {encoding: "UTF-8"},
*    encoding: "UTF-8",
*    properties: {},
*    mimeType: "application/json"
*  }
*
*
*
*
*  var Utils = "fun sum(a,b) = a +b"
*  output application/json
*  ---
*  {
*    "execute_ok" : run("main.dwl", {"main.dwl": "{a: 1}"}, {"payload": jsonValue }),
*    "logs" : do {
*      var execResult = run("main.dwl", {"main.dwl": "{a: log(1)}"}, {"payload": jsonValue })
*      ---
*      {
*          m: execResult.logs.message,
*          l: execResult.logs.level
*      }
*    },
*    "grant" : eval("main.dwl", {"main.dwl": "{a: readUrl(`http://google.com`)}"}, {"payload": jsonValue }, {},{ securityManager: (grant, args) -> false }),
*    "library" : eval("main.dwl", {"main.dwl": "Utils::sum(1,2)", "/Utils.dwl": Utils }, {"payload": jsonValue }),
*    "timeout" : eval("main.dwl", {"main.dwl": "(1 to 1000000000000) map \$ + 1" }, {"payload": jsonValue }, {},{timeOut: 2}).success,
*    "execFail" : eval("main.dwl", {"main.dwl": "dw::Runtime::fail('My Bad')" }, {"payload": jsonValue }),
*    "parseFail" : eval("main.dwl", {"main.dwl": "(1 + " }, {"payload": jsonValue }),
*    "writerFail" : eval("main.dwl", {"main.dwl": "output application/xml --- 2" }, {"payload": jsonValue }),
*    "defaultOutput" : eval("main.dwl", {"main.dwl": "payload" }, {"payload": jsonValue2}, {},{outputMimeType: "application/csv", writerProperties: {"separator": "|"}}),
*    "onExceptionFail": do  {
*      dw::Runtime::try( () ->
*          eval("main.dwl", {"main.dwl": "dw::Runtime::fail('Failing Test')" }, {"payload": jsonValue2}, {},{onException: "FAIL"})
*      ).success
*    },
*    "customLogger":
*         eval(
*   "main.dwl",
*            {"main.dwl": "log(1234)" },
*    {"payload": jsonValue2},
*     {},
*    {
*                   loggerService: {
*                      initialize: () -> {token: "123"},
*                      log: (level, msg, context) -> log("$(level) $(msg)", context)
*                   }
*                 }
*            )
*  }
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "execute_ok": {
*     "success": true,
*     "value": "{\n  a: 1\n}",
*     "mimeType": "application/dw",
*     "encoding": "UTF-8",
*     "logs": [
*
*     ]
*   },
*   "logs": {
*     "m": [
*       "1"
*     ],
*     "l": [
*       "INFO"
*     ]
*   },
*   "grant": {
*     "success": false,
*     "message": "The given required permissions: `Resource` are not being granted for this execution.\nTrace:\n  at readUrl (Unknown)\n  at main::main (line: 1, column: 5)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "end": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "content": "Unknown location"
*     },
*     "stack": [
*       "readUrl (anonymous:0:0)",
*       "main (main:1:5)"
*     ],
*     "logs": [
*
*     ]
*   },
*   "library": {
*     "success": true,
*     "value": 3,
*     "logs": [
*
*     ]
*   },
*   "timeout": true,
*   "execFail": {
*     "success": false,
*     "message": "My Bad\nTrace:\n  at fail (Unknown)\n  at main::main (line: 1, column: 1)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "end": {
*         "index": 0,
*         "line": 0,
*         "column": 0
*       },
*       "content": "Unknown location"
*     },
*     "stack": [
*       "fail (anonymous:0:0)",
*       "main (main:1:1)"
*     ],
*     "logs": [
*
*     ]
*   },
*   "parseFail": {
*     "success": false,
*     "message": "Invalid input \"1 + \", expected parameter or parenEnd (line 1, column 2):\n\n\n1| (1 + \n    ^^^^\nLocation:\nmain (line: 1, column:2)",
*     "location": {
*       "start": {
*         "index": 0,
*         "line": 1,
*         "column": 2
*       },
*       "end": {
*         "index": 4,
*         "line": 1,
*         "column": 6
*       },
*       "content": "\n1| (1 + \n    ^^^^"
*     },
*     "logs": [
*
*     ]
*   },
*   "writerFail": {
*     "success": true,
*     "value": 2,
*     "logs": [
*
*     ]
*   },
*   "defaultOutput": {
*     "success": true,
*     "value": {
*       "name": "Mariano",
*       "lastName": "achaval"
*     },
*     "logs": [
*
*     ]
*   },
*   "onExceptionFail": false,
*   "customLogger": {
*     "success": true,
*     "value": 1234,
*     "logs": [
*
*     ]
*   }
* }
*
*
* ----
**/
@Experimental()
@RuntimePrivilege(requires = "Execution")
fun eval(fileToExecute: String, fs: Dictionary<String>, readerInputs: Dictionary<ReaderInput> = {}, inputValues: Dictionary<Any> = {}, configuration: RuntimeExecutionConfiguration = {}): EvalSuccess | ExecutionFailure =
      native("system::EvalScriptFunctionValue")

/**
* *EXPERIMENTAL*
*
* Runs the script under the specified url.
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | url | The file name to execute
* | readerInputs | The inputs that are going to be read and bind to the execution
* | inputValues | The inputs that are going to be bind directly to the execution
* | configuration | The runtime configuration.
* |===
*
* === Example
*
* This example shows how the `runUrl` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::Runtime
* var jsonValue = {
*   value: '{"name": "Mariano"}' as Binary {encoding: "UTF-8"},
*   encoding: "UTF-8",
*   properties: {},
*   mimeType: "application/json"
* }
*
* var Utils = "fun sum(a,b) = a +b"
* ---
* {
*   "execute_ok" : runUrl("classpath://org/mule/weave/v2/engine/runtime_runUrl/example.dwl", {"payload": jsonValue })
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "execute_ok": {
*      "success": true,
*      "value": "\"Mariano\"",
*      "mimeType": "application/dw",
*      "encoding": "UTF-8",
*      "logs": [
*
*      ]
*    }
*  }
* ----
**/
@Experimental()
@RuntimePrivilege(requires = "Execution")
fun runUrl(url: String, readerInputs: Dictionary<ReaderInput> = {}, inputValues: Dictionary<Any> = {}, configuration: RuntimeExecutionConfiguration = {}): RunSuccess | ExecutionFailure = do {
    run("Main.dwl",{"Main.dwl": readUrl(url, "text/plain") as String {encoding: "UTF-8"}}, readerInputs , inputValues,configuration)
}

/**
* *EXPERIMENTAL*
*
* Evaluates the script under the specified url.
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | url | The file name to execute
* | readerInputs | The inputs that are going to be read and bind to the execution
* | inputValues | The inputs that are going to be bind directly to the execution
* | configuration | The runtime configuration.
* |===
*
* === Example
*
* This example shows how the `evalUrl` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var jsonValue = {
*   value: '{"name": "Mariano"}' as Binary {encoding: "UTF-8"},
*   encoding: "UTF-8",
*   properties: {},
*   mimeType: "application/json"
* }
*
* var Utils = "fun sum(a,b) = a +b"
* output application/json
* ---
* {
*   "execute_ok" : evalUrl("classpath://org/mule/weave/v2/engine/runtime_evalUrl/example.dwl", {"payload": jsonValue }),
*   "execute_ok_withValue" : evalUrl("classpath://org/mule/weave/v2/engine/runtime_evalUrl/example.dwl", {}, {"payload": {name: "Mariano"}})
* }
* ----
*
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "execute_ok": {
*      "success": true,
*      "value": "Mariano",
*      "logs": [
*
*      ]
*    },
*    "execute_ok_withValue": {
*      "success": true,
*      "value": "Mariano",
*      "logs": [
*
*      ]
*    }
*  }
* ----
**/
@Experimental()
@RuntimePrivilege(requires = "Execution")
fun evalUrl(url: String, readerInputs: Dictionary<ReaderInput> = {}, inputValues: Dictionary<Any> = {}, configuration: RuntimeExecutionConfiguration = {}): EvalSuccess | ExecutionFailure = do {
    eval("Main.dwl",{"Main.dwl": readUrl(url, "text/plain") as String {encoding: "UTF-8"}}, readerInputs , inputValues,configuration)
}

/**
* Returns the description of all the DataFormats that are installed in the current Runtime
*
*
*
* === Example
*
* This example shows how the `dataFormatsDescriptor` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::Runtime
* ---
* dataFormatsDescriptor()
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*   [
*         {
*              "id": "json",
*              "binary": false,
*              "defaultEncoding": "UTF-8",
*              "extensions": [
*                ".json"
*              ],
*              "defaultMimeType": "application/json",
*              "acceptedMimeTypes": [
*                "application/json"
*              ],
*              "readerProperties": [
*                {
*                  "name": "streaming",
*                  "optional": true,
*                  "defaultValue": false,
*                  "description": "Used for streaming input (use only if entries are accessed sequentially).",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                }
*              ],
*              "writerProperties": [
*                {
*                  "name": "writeAttributes",
*                  "optional": true,
*                  "defaultValue": false,
*                  "description": "Indicates that if a key has attributes, they are going to be added as children key-value pairs of the key that contains them. The attribute new key name will start with @.",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                },
*                {
*                  "name": "skipNullOn",
*                  "optional": true,
*                  "defaultValue": "None",
*                  "description": "Indicates where is should skips null values if any or not. By default it doesn't skip.",
*                  "possibleValues": [
*                    "arrays",
*                    "objects",
*                    "everywhere"
*                  ]
*                }
*              ]
*            },
*            {
*              "id": "xml",
*              "binary": false,
*              "extensions": [
*                ".xml"
*              ],
*              "defaultMimeType": "application/xml",
*              "acceptedMimeTypes": [
*                "application/xml"
*              ],
*              "readerProperties": [
*                {
*                  "name": "supportDtd",
*                  "optional": true,
*                  "defaultValue": true,
*                  "description": "Whether DTD handling is enabled or disabled; disabling means both internal and external subsets will just be skipped unprocessed.",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                },
*                {
*                  "name": "streaming",
*                  "optional": true,
*                  "defaultValue": false,
*                  "description": "Used for streaming input (use only if entries are accessed sequentially).",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                },
*                {
*                  "name": "maxEntityCount",
*                  "optional": true,
*                  "defaultValue": 1,
*                  "description": "The maximum number of entity expansions. The limit is in place to avoid Billion Laughs attacks.",
*                  "possibleValues": [
*
*                  ]
*                }
*              ],
*              "writerProperties": [
*                {
*                  "name": "writeDeclaration",
*                  "optional": true,
*                  "defaultValue": true,
*                  "description": "Indicates whether to write the XML header declaration or not.",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                },
*                {
*                  "name": "indent",
*                  "optional": true,
*                  "defaultValue": true,
*                  "description": "Indicates whether to indent the code for better readability or to compress it into a single line.",
*                  "possibleValues": [
*                    true,
*                    false
*                  ]
*                }
*              ]
*            }
*   ]
* ----
**/
@Experimental()
fun dataFormatsDescriptor(): Array<DataFormatDescriptor> = native("system::DataFormatDescriptorsFunctionValue")

/**
 * Stops the execution for the specified timeout (in milliseconds).
 *
 *
 * WARNING: Stopping the execution will block the thread used, potentially
 * causing slowness, low performance and potentially freezes of the entire
 * runtime. This operation is intended for limited functional testing purposes
 * and should not be used in production application, performance testing or
 * with multiple applications deployed.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `value` | Input of any type.
 * | `timeout` | The number of milliseconds to wait.
 * |===
 *
 * === Example
 *
 * This example waits 2000 milliseconds (2 seconds) to execute.
 *
 * ==== Source
 *
 * [source,DataWeave, linenums]
 * ----
 * %dw 2.0
 * import * from dw::Runtime
 * output application/json
 * ---
 * { "user" : 1 } wait 2000
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "user": 1 }
 * ----
 */
fun wait <T>(value: T, timeout: Number): T = native("system::wait")


// A type: `TryResult`.
/**
 * Object with a result or error message. If `success` is `false`, it contains
 * the `error`. If `true`, it provides the `result`.
 *
 */
type TryResult<T> = {
  success: Boolean,
  result?: T,
  error?: {
    kind: String,
    message: String,
    stack?: Array<String>,
    /**
    * Since 4.4.0. It is only available when stack is not present. It will contain the native java stacktrace.
    */
    stackTrace?: String,
    location?: String
  }
}

/**
 * Evaluates the delegate function and returns an object with `success: true` and `result` if the delegate function succeeds, or an object with `success: false` and `error` if the delegate function throws an exception.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `delegate` | The function to evaluate.
 * |===
 *
 * === Example
 *
 * This example calls the `try` function using the `randomNumber` function as argument. +
 * The function `randomNumber` generates a random number and calls `fail` if the number is higher than 0.5. The declaration of this function is in the script's header.
 *
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import try, fail from dw::Runtime
 * output application/json
 * fun randomNumber() =
 * if(random() > 0.5)
 *   fail("This function is failing")
 *  else
 *   "OK"
 * ---
 * try(() -> randomNumber())
 * ----
 *
 * ==== Output
 *
 * When `randomNumber` fails, the output is:
 * [source,JSON,linenums]
 * ----
 * {
 *   "success": false,
 *   "error": {
 *     "kind": "UserException",
 *     "message": "This function is failing",
 *     "location": "Unknown location",
 *     "stack": [
 *       "fail (anonymous:0:0)",
 *       "myFunction (anonymous:1:114)",
 *       "main (anonymous:1:179)"
 *     ]
 *   }
 * }
 * ----
 *
 * When `randomNumber` succeeds, the output is:
 * [source,JSON,linenums]
 * ----
 * {
 *   "success": true,
 *   "result": "OK"
 * }
 * ----
 * The `orElseTry` and `orElse` functions will also continue processing if the `try` function fails. See the `orElseTry` and `orElse` documentation for more complete examples of handling failing `try` function expressions.
 *
 * Note: Instead of using the `orElseTry` and `orElse` functions, based on the output of the `try` function, you can add conditional logic to execute when the result is `success: true` or `success: false`.
 */
fun try<T>(delegate: () -> T): TryResult<T> = native("system::try")

/**
* Function to be use with try in order to chain multiple `try`
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `previous` | Previous try result
* | `orElse` | The next option to try if the previous fails
* |===
*
* === Example
*
* This example waits shows how to chain different try
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var user = {}
* var otherUser = {}
* output application/json
* ---
* {
*     a: try(() -> user.name!) orElseTry otherUser.name!,
*     b: try(() -> user.name!) orElseTry "No User Name"
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": {
*     "success": false,
*     "error": {
*       "kind": "KeyNotFoundException",
*       "message": "There is no key named 'name'",
*       "location": "\n9|     a: try(() -> user.name!) orElseTry otherUser.name!,\n                                          ^^^^^^^^^^^^^^",
*       "stack": [
*         "main (org::mule::weave::v2::engine::transform:9:40)"
*       ]
*     }
*   },
*   "b": {
*     "success": true,
*     "result": "No User Name"
*   }
* }
* ----
*/
@Since(version = "2.2.0")
fun orElseTry<T, R>(previous: TryResult<T>, orElse: () -> R): TryResult<T | R> = do {
    if(previous.success)
       previous
    else
       try(orElse)
}

/**
* Returns the result of the `orElse` if the `previous` try result failed if not returns the result of the `previous`
*
*
* _Introduced in DataWeave 2.2.0. Supported by Mule 4.2 and later._
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `previous` | Previous try result
* | `orElse` | The next option to try if the previous fails
* |===
*
* === Example
*
* This example waits shows how to chain different try
*
* ==== Source
*
* [source,DataWeave, linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var user = {}
* var otherUser = {name: "DW"}
* output application/json
* ---
* {
*     a: try(() -> user.name!) orElse "No User Name",
*     b: try(() -> otherUser.name) orElse "No User Name"
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "No User Name",
*   "b": "DW"
* }
* ----
*/
@Since(version = "2.2.0")
fun orElse<T, R>(previous: TryResult<T>, orElse: () -> R): T | R = do {
    if(previous.success)
       previous.result!
    else
       orElse()
}

/**
* Returns the location string of a given value.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `value` | A value of any type.
* |===
*
* === Example
*
* This example returns the contents of the line (the location) that defines
* variable `a` in the header of the DataWeave script.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* var a = 123
* output application/json
* ---
* locationString(a)
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* "var a = 123"
* ----
*/
fun locationString(value:Any): String = native("system::locationString")


/**
* Returns all the properties configured for Mule runtime.
*
*
* === Example
*
* This example returns all properties from the `java.util.Properties` class.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* output application/dw
* ---
* { "props" : props() }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*  props: {
*    "java.vendor": "Oracle Corporation" as String {class: "java.lang.String"},
*    "sun.java.launcher": "SUN_STANDARD" as String {class: "java.lang.String"},
*    "sun.management.compiler": "HotSpot 64-Bit Tiered Compilers" as String ..., *    "os.name": "Mac OS X" as String {class: "java.lang.String"},
*    "sun.boot.class.path": "/Library/Java/JavaVirtualMachines/ ...,
*    "org.glassfish.grizzly.nio.transport.TCPNIOTransport...": "1048576" ...,
*    "java.vm.specification.vendor": "Oracle Corporation" as String ...,
*    "java.runtime.version": "1.8.0_111-b14" as String {class: "java.lang.String"},
*    "wrapper.native_library": "wrapper" as String {class: "java.lang.String"},
*    "wrapper.key": "XlIl4YartmfEU3oKu7o81kNQbwhveXi-" as String ...,
*    "user.name": "me" as String {class: "java.lang.String"},
*    "mvel2.disable.jit": "TRUE" as String {class: "java.lang.String"},
*    "user.language": "en" as String {class: "java.lang.String"} ...,
*    "sun.boot.library.path": "/Library/Java/JavaVirtualMachines ...
*    "xpath.provider": "com.mulesoft.licm.DefaultXPathProvider" ...,
*    "wrapper.backend": "pipe" as String {class: "java.lang.String"},
*    "java.version": "1.8.0_111" as String {class: "java.lang.String"},
*    "user.timezone": "America/Los_Angeles" as String {class: "java.lang.String"},
*    "java.net.preferIPv4Stack": "TRUE" as String {class: "java.lang.String"},
*    "sun.arch.data.model": "64" as String {class: "java.lang.String"},
*    "java.endorsed.dirs": "/Library/Java/JavaVirtualMachines/...,
*    "sun.cpu.isalist": "" as String {class: "java.lang.String"},
*    "sun.jnu.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "mule.testingMode": "" as String {class: "java.lang.String"},
*    "file.encoding.pkg": "sun.io" as String {class: "java.lang.String"},
*    "file.separator": "/" as String {class: "java.lang.String"},
*    "java.specification.name": "Java Platform API Specification" ...,
*    "java.class.version": "52.0" as String {class: "java.lang.String"},
*    "jetty.git.hash": "82b8fb23f757335bb3329d540ce37a2a2615f0a8" ...,
*    "user.country": "US" as String {class: "java.lang.String"},
*    "mule.agent.configuration.folder": "/Applications/AnypointStudio.app/ ...,
*    "log4j.configurationFactory": "org.apache.logging.log4j.core...",
*    "java.home": "/Library/Java/JavaVirtualMachines/...,
*    "java.vm.info": "mixed mode" as String {class: "java.lang.String"},
*    "wrapper.version": "3.5.34-st" as String {class: "java.lang.String"},
*    "os.version": "10.13.4" as String {class: "java.lang.String"},
*    "org.eclipse.jetty.LEVEL": "WARN" as String {class: "java.lang.String"},
*    "path.separator": ":" as String {class: "java.lang.String"},
*    "java.vm.version": "25.111-b14" as String {class: "java.lang.String"},
*    "wrapper.pid": "5212" as String {class: "java.lang.String"},
*    "java.util.prefs.PreferencesFactory": "com.mulesoft.licm..."},
*    "wrapper.java.pid": "5213" as String {class: "java.lang.String"},
*    "mule.home": "/Applications/AnypointStudio.app/...,
*    "java.awt.printerjob": "sun.lwawt.macosx.CPrinterJob" ...,
*    "sun.io.unicode.encoding": "UnicodeBig" as String {class: "java.lang.String"},
*    "awt.toolkit": "sun.lwawt.macosx.LWCToolkit" ...,
*    "org.glassfish.grizzly.nio.transport...": "1048576" ...,
*    "user.home": "/Users/me" as String {class: "java.lang.String"},
*    "java.specification.vendor": "Oracle Corporation" ...,
*    "java.library.path": "/Applications/AnypointStudio.app/...,
*    "java.vendor.url": "http://java.oracle.com/" as String ...,
*    "java.vm.vendor": "Oracle Corporation" as String {class: "java.lang.String"},
*    gopherProxySet: "false" as String {class: "java.lang.String"},
*    "wrapper.jvmid": "1" as String {class: "java.lang.String"},
*    "java.runtime.name": "Java(TM) SE Runtime Environment" ...,
*    "mule.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "sun.java.command": "org.mule.runtime.module.reboot....",
*    "java.class.path": "%MULE_LIB%:/Applications/AnypointStudio.app...",
*    "log4j2.loggerContextFactory": "org.mule.runtime.module.launcher...,
*    "java.vm.specification.name": "Java Virtual Machine Specification" ,
*    "java.vm.specification.version": "1.8" as String {class: "java.lang.String"},
*    "sun.cpu.endian": "little" as String {class: "java.lang.String"},
*    "sun.os.patch.level": "unknown" as String {class: "java.lang.String"},
*    "com.ning.http.client.AsyncHttpClientConfig.useProxyProperties": "true" ...,
*    "wrapper.cpu.timeout": "10" as String {class: "java.lang.String"},
*    "java.io.tmpdir": "/var/folders/42/dd73l3rx7qz0n625hr29kty80000gn/T/" ...,
*    "anypoint.platform.analytics_base_uri": ...,
*    "java.vendor.url.bug": "http://bugreport.sun.com/bugreport/" ...,
*    "os.arch": "x86_64" as String {class: "java.lang.String"},
*    "java.awt.graphicsenv": "sun.awt.CGraphicsEnvironment" ...,
*    "mule.base": "/Applications/AnypointStudio.app...",
*    "java.ext.dirs": "/Users/staceyduke/Library/Java/Extensions: ..."},
*    "user.dir": "/Applications/AnypointStudio.app/..."},
*    "line.separator": "\n" as String {class: "java.lang.String"},
*    "java.vm.name": "Java HotSpot(TM) 64-Bit Server VM" ...,
*    "org.quartz.scheduler.skipUpdateCheck": "true" ...,
*    "file.encoding": "UTF-8" as String {class: "java.lang.String"},
*    "mule.forceConsoleLog": "" as String {class: "java.lang.String"},
*    "java.specification.version": "1.8" as String {class: "java.lang.String"},
*    "wrapper.arch": "universal" as String {class: "java.lang.String"}
*  } as Object {class: "java.util.Properties"}
* ----
*/
@RuntimePrivilege(requires = "Properties")
fun props(): Dictionary<String> = native("system::props")

/**
* Returns the value of the property with the specified name or `null` if the
* property is not defined.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `propertyName` | The property to retrieve.
* |===
*
* === Example
*
* This example gets the `user.timezone` property.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::Runtime
* output application/dw
* ---
* { "props" : prop("user.timezone") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { props: "America/Los_Angeles" as String {class: "java.lang.String"} }
* ----
*/
@RuntimePrivilege(requires = "Properties")
fun prop(propertyName: String): String | Null = props()[propertyName]
