/**
* This module contains functions for working with and creating Period values.
*
* To use this module, you must import it to your DataWeave code,
* for example, by adding the line `import * from dw::core::Periods`
* to the header of your DataWeave script.
*/
%dw 2.0

/**
*
* Returns a Period (P) value consisting of the number
* of years, months, and days between two Date values.
*
*
* The start date is included, but the end date is not.
* The result of this method can be a negative period
* if the end date (`endDateExclusive`) is before the
* start date (`startDateInclusive`).
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `startDateInclusive` | The start date, inclusive.
* | `endDateExclusive` | The end date, exclusive.
* |===
*
* === Example
*
* This example shows how `between` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::core::Periods
* output application/json
* ---
* {
*    a: between(|2010-12-12|,|2010-12-10|),
*    b: between(|2010-11-10|,|2011-12-11|),
*    c: between(|2020-02-29|,|2020-03-30|)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": "P2D",
*    "b": "P-1Y-1M-1D",
*    "c": "P-1M-1D"
*  }
* ----
**/
@Since(version = "2.4.0")
fun between(startDateInclusive: Date, endDateExclusive: Date): Period = native("system::BetweenLocalDateOperator")


/**
* Creates a Period value that represents a number of days, hours,
* minutes, or seconds.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `period` | An object such as `{days:4, hours:11, minutes:45, seconds: 55}`.
*              The key-value pairs are optional and can be specified in any
*              order. An empty object (`{}`) returns the Period value
*              `"PT0S"` (zero seconds). The default value of each key
*              is `0`. Valid values are whole or decimal numbers, which
*              can be positive or negative. Key names are selectable.
* |===
*
* === Example
*
* This example shows how `duration` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*    dayAfterDateTime: |2020-10-05T20:22:34.385Z| + duration({days: 1}),
*    dayAndHourBeforeDateTime: |2020-10-05T20:22:34.385Z| - duration({days: 1, hours: 1}),
*    pointInTimeBefore: |2020-10-05T20:22:34.385Z| - duration({days: 1, hours: 1, minutes: 20, seconds: 10}),
*    emptyDuration: duration({}),
*    constructDuration: duration({days:4, hours:11, minutes:28}),
*    selectHoursFromDuration: duration({days:4, hours:11, minutes:28}).hours,
*    decimalAsPeriod:  duration({seconds: 30.5}),
*    addNegativeValue: duration({ minutes : 1 }) + duration({ seconds : -1 })
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "dayAfterDateTime": "2020-10-06T20:22:34.385Z",
*    "dayAndHourBeforeDateTime": "2020-10-04T19:22:34.385Z",
*    "pointInTimeBefore": "2020-10-04T19:02:24.385Z",
*    "emptyDuration": "PT0S",
*    "constructDuration": "PT107H28M",
*    "selectHoursFromDuration": 11,
*    "decimalAsPeriod": "PT30.5S",
*    "addNegativeValue": 59
* }
* ----
**/
@Since(version = "2.4.0")
fun duration(period: {days?: Number, hours?: Number, minutes?: Number, seconds?: Number }): Period =  do {
    var nDays   = period.days default 0
    var days    = floor(nDays)
    var nHours  = period.hours default 0 + ((nDays - days) * 24)
    var hours   = floor(nHours)
    var nMinutes= period.minutes default 0 + ((nHours - hours) * 60)
    var minutes = floor(nMinutes )
    var seconds = period.seconds default 0 + ((nMinutes - minutes) * 60)
    ---
    "P$(days as String {format:'#'})DT$(hours as String {format:'#'})H$(minutes as String {format:'#'})M$(seconds as String)S" as Period
}

/**
* Creates a Period value as a date-based number of years, months,
* and days in the ISO-8601 calendar system.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `period` | An object such as `{years:4, months:11, days:28}`. The
*              key-value pairs are optional and can be specified in any
*              order. An empty object (`{}`) returns the Period value
*              `"P0D"` (zero days). The default value of each key
*              is `0`. Valid values are whole numbers, which can be
*              positive or negative. Decimal values produce an
*              error message. Key names are selectable.
* |===
*
* === Example
*
* This example shows how `period` behaves with different inputs. The example
* add a subtracts and adds the result of a `period` function to DateTime and
* Date values. It also constructs a Period value from `period` objects
* and selects a `months` value from the object.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from dw::core::Periods
* ---
* {
*    dayBeforeDateTime: |2020-10-05T20:22:34.385Z| - period({days:1}),
*    dayAfterDate: |2020-10-05| + period({days:1}),
*    yearMonthDayAfterDate: |2020-10-05| + period({years:1, months:1, days:1}),
*    emptyPeriod: period({}),
*    constructPeriod: period({years:4, months:11, days:28}),
*    selectMonthsFromPeriod: period({years:4, months:11, days:28}).months
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*     "dayBeforeDateTime": "2020-10-04T20:22:34.385Z",
*     "dayAfterDate": "2020-10-06",
*     "yearMonthDayAfterDate": "2021-11-06",
*     "emptyPeriod": "P0D",
*     "constructPeriod": "P4Y11M28D",
*     "selectMonthsFromPeriod": 11
* }
* ----
**/
@Since(version = "2.4.0")
fun period(period: {years?: Number, months?: Number, days?: Number}): Period =  do {
    var years = (period.years default 0)
    var months = (period.months default 0)
    var days = (period.days default 0)
    var nYears = if(isDecimal(years)) dw::Runtime::fail("Field years: `$(years)`, can not be decimal.") else years as String {format: "#"}
    var nMonth = if(isDecimal(months)) dw::Runtime::fail("Field months: `$(months)`, can not be decimal.") else months as String {format: "#"}
    var nDays = if(isDecimal(days)) dw::Runtime::fail("Field days: `$(days)`, can not be decimal.") else days as String {format: "#"}
    ---
    "P$(nYears)Y$(nMonth)M$(nDays)D" as Period
}

/**
* Creates a Period value from the provided number of years.
*
*
* The function applies the `period` function to the input value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nYears` | A whole number for the number of years.
*              A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `years` behaves with different inputs.
* It adds a year to a DateTime value, and it converts the whole
* number 4 into a number of years in the Period format (`P4Y`).
* Notice that `years(-1) + years(2)` returns the number of months.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*   nextYear: |2020-10-05T20:22:34.385Z| + years(1),
*   fourYearPeriod: years(4),
*   addNegativeValue: years(-1) + years(2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "nextYear": "2021-10-05T20:22:34.385Z",
*    "fourYearPeriod": "P4Y",
*    "addNegativeValue": 12
* }
* ----
**/
@Since(version = "2.4.0")
fun years(nYears: Number):Period =
  period({years: nYears})


/**
* Creates a Period value from the provided number of months.
*
*
* The function applies the `period` function to the input value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nMonths` | The number of months as a whole number.
*               A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `months` behaves with different inputs.
* It adds a month to a DateTime value, and it converts the
* whole number `4` to a number of months in the Period format (`P4M`).
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*   nextMonth: |2020-10-05T20:22:34.385Z| + months(1),
*   fourMonthPeriod : months(4),
*   addNegativeValue: months(-1) + months(2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "nextMonth": "2020-11-05T20:22:34.385Z",
*   "fourMonthPeriod": "P4M",
*   "addNegativeValue": 1
* }
* ----
**/
@Since(version = "2.4.0")
fun months(nMonths: Number):Period =
  period({months: nMonths})

/**
* Creates a Period value from the provided number of days.
*
*
* The function applies the `period` function to input that is a whole number
* and the `duration` function to decimal input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nDays` | The number of hours as a whole or decimal number.
*             A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `days` behaves with different inputs. It
* adds and subtracts hours from DateTime values. It also converts the
* decimal value `4.555` into a number of hours, minutes, and second in
* the Period format (`PT109H19M12S`) and the whole number `4` into a
* number of days in the Period format (`P4D`).
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*    tomorrow: |2020-10-05T20:22:34.385Z| + days(1),
*    yesterday: |2020-10-05T20:22:34.385Z| - days(1),
*    decimalDaysPlusQuarter:  |2020-10-05T00:00:00.000Z| + days(0.25),
*    decimalDaysPlusHalf:  |2020-10-05T00:00:00.000Z| + days(0.5),
*    decimalDaysPlusThreeQuarters:  |2020-10-05T00:00:00.000Z| + days(0.75),
*    decimalInputAsPeriod : days(4.555),
*    fourDayPeriod: days(4),
*    negativeValue: days(-1)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "tomorrow": "2020-10-06T20:22:34.385Z",
*    "yesterday": "2020-10-04T20:22:34.385Z",
*    "decimalDaysPlusQuarter": "2020-10-05T06:00:00Z",
*    "decimalDaysPlusHalf": "2020-10-05T12:00:00Z",
*    "decimalDaysPlusThreeQuarters": "2020-10-05T18:00:00Z",
*    "decimalInputAsPeriod": "PT109H19M12S",
*    "fourDayPeriod": "P4D",
*    "negativeValue": "P-1D"
* }
* ----
**/
@Since(version = "2.4.0")
fun days(nDays: Number): Period =
    if(isDecimal(nDays))
        duration({days: nDays})
    else
        period({days: nDays})

/**
* Creates a Period value from the provided number of hours.
*
*
* The function applies the `duration` function to the input value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nHours` | The number of hours as a whole or decimal number.
*              A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `hours` behaves with different inputs.
* It adds and subtracts hours from DateTime and LocalTime values.
* It also converts the decimal value `4.555` into the Period format
* (`PT4H33M18S`) and the whole number `4` into the Period format (`PT4H`).
* Notice that `hours(-1) + hours(2)` returns the number of seconds.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*    nextHour: |2020-10-05T20:22:34.385Z| + hours(1),
*    previousHour: |2020-10-05T20:22:34.385Z| - hours(1),
*    threeHoursLater: |20:22| + hours(3),
*    addDecimalInput: |20:22| + hours(3.5),
*    decimalInputAsPeriod : hours(4.555),
*    fourHourPeriod : hours(4),
*    addNegativeValue: hours(-1) + hours(2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "nextHour": "2020-10-05T21:22:34.385Z",
*    "previousHour": "2020-10-05T19:22:34.385Z",
*    "threeHoursLater": "23:22:00",
*    "addDecimalInput": "23:52:00",
*    "decimalInputAsPeriod": "PT4H33M18S",
*    "fourHourPeriod": "PT4H",
*    "addNegativeValue": 3600
* }
* ----
**/
@Since(version = "2.4.0")
fun hours(nHours: Number):Period =
  duration({hours: nHours})



/**
* Creates a Period value from the provided number of minutes.
*
*
* The function applies the `duration` function to the input value.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nMinutes` | The number of minutes as a whole or decimal number.
*                A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `minutes` behaves with different inputs.
* It adds and subtracts hours from DateTime values. It also converts
* the decimal value `4.555` into the Period format (`PT4M33.3S`) and
* and the whole number `4` into the Period format (`PT4M`). Notice
* that `minutes(-1) + minutes(2)` returns the number of seconds.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*    nextMinute: |2020-10-05T20:22:34.385Z| + minutes(1),
*    previousMinute: |2020-10-05T20:22:34.385Z| - minutes(1),
*    decimalInputPeriod: minutes(4.555),
*    wholeNumberInputPeriod: minutes(4),
*    addNegativeValue: minutes(-1) + minutes(2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "nextMinute": "2020-10-05T20:23:34.385Z",
*    "previousMinute": "2020-10-05T20:21:34.385Z",
*    "decimalInputPeriod": "PT4M33.3S",
*    "wholeNumberInputPeriod": "PT4M",
*    "addNegativeValue": 60
* }
* ----
**/
@Since(version = "2.4.0")
fun minutes(nMinutes: Number):Period =
  duration({minutes: nMinutes})



/**
* Creates a Period value from the provided number of seconds.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `nSecs` | The number of seconds as a whole or decimal number.
*             A positive or negative number is valid.
* |===
*
* === Example
*
* This example shows how `seconds` behaves with different inputs.
* It adds and subtracts seconds from DateTime values. It also converts
* the decimal value `4.555` into the Period format (`PT4.555S`) and
* and the whole number `4` into the Period format (`PT4S`)
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Periods
* output application/json
* ---
* {
*   nextSecond: |2020-10-05T20:22:34.385Z| + seconds(1),
*   previousSecond: |2020-10-05T20:22:34.385Z| - seconds(1),
*   decimalInputPeriod: seconds(4.555),
*   wholeNumberInputPeriod: seconds(4),
*   addNegativeValue: seconds(-1) + seconds(2)
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "nextSecond": "2020-10-05T20:22:35.385Z",
*   "previousSecond": "2020-10-05T20:22:33.385Z",
*   "decimalInputPeriod": "PT4.555S",
*   "wholeNumberInputPeriod": "PT4S",
*   "addNegativeValue": 1
* }
* ----
**/
@Since(version = "2.4.0")
fun seconds(nSecs: Number):Period =
  duration({seconds: nSecs})
