/**
* This module provides resources for registering a new data
* format for the DataWeave language.
*
* For an example, see
https://github.com/mulesoft-labs/data-weave-custom-data-format[Custom Data Formats Example].
*
*/
%dw 2.0

/**
* Registration hook that the DataWeave engine uses to discover the variable that represents the custom data format. For an example, see the https://github.com/mulesoft-labs/data-weave-custom-data-format/blob/master/README.md[Custom Data Formats Example README].
*/
@AnnotationTarget(targets = ["Variable"])
annotation DataFormatExtension()

/**
 * Represents a MIME type, such as `application/json`.
 */
@Since(version = "2.2.0")
type MimeType = String

/**
 * Reader or writer configuration settings.
 */
@Since(version = "2.2.0")
type Settings = {}

/**
 * Represents a configuration with no settings.
 */
@Since(version = "2.2.0")
type EmptySettings = {}

/**
 * Represents encoding settings.
 */
@Since(version = "2.2.0")
type EncodingSettings = {
    /**
    * Encoding that the writer uses for output. Defaults to "UTF-8".
    */
    encoding?: String {defaultValue: "UTF-8"}
}

/**
 * Represents the DataFormat definition.
 */
@Since(version = "2.2.0")
type DataFormat<ReaderSettings <: Settings, WriterSettings <: Settings> = {
    /**
    * True if this is data format is represented as binary representation instead of text. False if not present.
    */
    binaryFormat?: Boolean,
    /**
    * Default character set of this format, if any.
    */
    defaultCharset?: String,

    /**
    * Returns the list of file extensions with the `.` (for example, `.json`, `.xml`) to assign to this data format.
    */
    fileExtensions?: Array<String>,

    /**
    * The list of MIME types to accept.
    */
    acceptedMimeTypes: Array<MimeType>,

    /**
    * Function that reads raw content and transforms it into the canonical DataWeave model.
    */
    reader: (content: Binary, charset: String, settings: ReaderSettings) -> Any,

    /**
    * Function that writes the canonical DataWeave model into binary content.
    */
    writer: (value: Any, settings: WriterSettings) -> Binary
}
