/**
* Generates test report based on https://docs.sonarqube.org/latest/analysis/generic-test/
*/
%dw 2.0
import countBy from dw::core::Arrays
import * from dw::test::internal::reporter::Reports
import TestResult from dw::test::Tests

input testResult: TestsResult application/json

type FileType =
  {|
    file @(path: String)*: TestCasesType
  |}

type TestCasesType = {|
                         testCase @(name: String, duration: Number)*?: {
                           skipped @(message?: String)?: String | Null,
                           error @(message?: String)?: String | Null,
                           failure @(message?: String)?: String | Null,
                         }
                       |}

type SonarResult =
  {|
    testExecutions @(version: 1) : FileType
  |}


/**
* Transforms the test result in a junit compatible test result report
*/
fun processTestResult(testResult: TestsResult): SonarResult = do {

    fun toFilePath(name: String) = "$(name replace '::' with '/').dwl"

    fun testToSonarResult(test: TestResult, name: String = ""): TestCasesType = do {
        {
          (
            test.tests map ((testCase) ->
              if(!isEmpty(testCase.tests))
                {
                  (
                    testCase.tests map ((testCase) ->
                        testToSonarResult(testCase, if(isEmpty(name)) testCase.name else name ++ "/" ++ testCase.name))
                  )
                }
              else
                {
                  testCase @( name: (if(isEmpty(name)) testCase.name else name ++ "/" ++ testCase.name), duration: testCase.time) : {
                      (error: testCase.errorMessage default "") if(testCase.status == "ERROR"),
                      (failure: testCase.errorMessage default "") if(testCase.status == "FAIL"),
                      (skipped: testCase.errorMessage default "") if(testCase.status == "SKIP")
                  }
                }
            )
          )

       }
    }
    ---
    {
      testExecutions @(version: 1) : {
        (testResult map (testSuite) -> {
            file @(path: toFilePath(testSuite.name)): testToSonarResult(testSuite.testResult)
         })
      }
    }
}


output application/xml
---

  processTestResult(testResult)